/* This file is part of the KDE project
   Copyright (C) 2003 Peter Simonsson <psn@linux.se>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "kiviooptionsdialog.h"
#include "kivio_view.h"
#include "kivio_doc.h"
#include "kivio_page.h"
#include "kivioglobal.h"
#include "kivio_grid_data.h"
#include "kivio_canvas.h"
#include "kivio_settings.h"
#include "kivio_config.h"

#include <tdelocale.h>
#include <KoApplication.h>
#include <kiconloader.h>
#include <kpushbutton.h>
#include <KoPageLayoutDia.h>
#include <kurlrequester.h>
#include <kcolorbutton.h>
#include <KoUnitWidgets.h>
#include <tdeglobal.h>
#include <kdebug.h>
#include <tdefontdialog.h>

#include <tqlabel.h>
#include <tqbuttongroup.h>
#include <tqgroupbox.h>
#include <tqcombobox.h>
#include <tqcheckbox.h>
#include <tqradiobutton.h>
#include <tqlayout.h>
#include <tqtabwidget.h>

// GuidesListViewItem::GuidesListViewItem(TQListView* parent, KivioGuideLineData *gd, KoUnit::Unit u)
// : TDEListViewItem(parent), m_data(gd)
// {
//   setPixmap(0, BarIcon(m_data->orientation() == TQt::Vertical ?
//     "guides_vertical":"guides_horizontal"));
//   TQString s = TDEGlobal::_locale->formatNumber(KoUnit::toUserValue(m_data->position(), u), 2);
//   s += " " + KoUnit::unitName(u);
//   setText(1, s);
// }
// 
// GuidesListViewItem::~GuidesListViewItem()
// {
//   delete m_data;
// }
// 
// void GuidesListViewItem::setUnit(KoUnit::Unit u)
// {
//   TQString s = TDEGlobal::_locale->formatNumber(KoUnit::toUserValue(m_data->position(), u), 2);
//   s += " " + KoUnit::unitName(u);
//   setText(1, s);
// }
// 
// void GuidesListViewItem::setPosition(double p, KoUnit::Unit u)
// {
//   m_data->setPosition(KoUnit::fromUserValue(p, u));
//   TQString s = TDEGlobal::_locale->formatNumber(p, 2);
//   s += " " + KoUnit::unitName(u);
//   setText(1, s);
// }
// 
// void GuidesListViewItem::setOrientation(TQt::Orientation o)
// {
//   m_data->setOrientation(o);
//   setPixmap(0, BarIcon(m_data->orientation() == TQt::Vertical ?
//     "guides_vertical":"guides_horizontal"));
// }

/*****************************************************************************/

KivioOptionsDialog::KivioOptionsDialog(KivioView* parent, const char* name)
  : KDialogBase(IconList, i18n("Settings"), Ok|Cancel|Apply|Default, Ok, parent, name)
{
  initPage();
  initGrid();
  unitChanged(parent->doc()->unit());
}

void KivioOptionsDialog::initPage()
{
  TQFrame* page = addPage(i18n("Page"), i18n("Page Settings"),
    tdeApp->iconLoader()->loadIcon("application-x-zerosize", TDEIcon::Toolbar, 32));
  m_pageIndex = pageIndex(page);

  KivioView* view = static_cast<KivioView*>(parent());
  KoUnit::Unit unit = KoUnit::unit(Kivio::Config::unit());
  m_layout = Kivio::Config::defaultPageLayout();
  m_font = Kivio::Config::font();

  TQLabel* unitLbl = new TQLabel(i18n("Default &units:"), page);
  m_unitCombo = new TQComboBox(page);
  m_unitCombo->insertStringList(KoUnit::listOfUnitName());
  m_unitCombo->setCurrentItem(unit);
  unitLbl->setBuddy(m_unitCombo);
  TQLabel* layoutLbl = new TQLabel(i18n("Default layout:"), page);
  m_layoutTxtLbl = new TQLabel(page);
  m_layoutTxtLbl->setFrameStyle(TQFrame::LineEditPanel | TQFrame::Sunken);
  m_layoutTxtLbl->setSizePolicy(TQSizePolicy(
    TQSizePolicy::Minimum, TQSizePolicy::Fixed));
  setLayoutText(m_layout);
  KPushButton* layoutBtn = new KPushButton(i18n("Change..."), page);
  layoutBtn->setSizePolicy(TQSizePolicy(
    TQSizePolicy::Fixed, TQSizePolicy::Fixed));
  TQLabel* fontLbl = new TQLabel(i18n("Default font:"), page);
  m_fontTxtLbl = new TQLabel(page);
  m_fontTxtLbl->setFrameStyle(TQFrame::LineEditPanel | TQFrame::Sunken);
  m_fontTxtLbl->setSizePolicy(TQSizePolicy(
    TQSizePolicy::Minimum, TQSizePolicy::Fixed));
  setFontText(m_font);
  KPushButton* fontBtn = new KPushButton(i18n("Change..."), page);
  layoutBtn->setSizePolicy(TQSizePolicy(
    TQSizePolicy::Fixed, TQSizePolicy::Fixed));
  m_marginsChBox = new TQCheckBox(i18n("Show page &margins"), page);
  m_marginsChBox->setChecked(view->isShowPageMargins());
  m_rulersChBox = new TQCheckBox(i18n("Show page &rulers"), page);
  m_rulersChBox->setChecked(view->isShowRulers());

  TQGridLayout* gl = new TQGridLayout(page);
  gl->setSpacing(KDialog::spacingHint());
  gl->addWidget(unitLbl, 0, 0);
  gl->addMultiCellWidget(m_unitCombo, 0, 0, 1, 2);
  gl->addWidget(layoutLbl, 1, 0);
  gl->addWidget(m_layoutTxtLbl, 1, 1);
  gl->addWidget(layoutBtn, 1, 2);
  gl->addWidget(fontLbl, 2, 0);
  gl->addWidget(m_fontTxtLbl, 2, 1);
  gl->addWidget(fontBtn, 2, 2);
  gl->addMultiCellWidget(m_marginsChBox, 3, 3, 0, 2);
  gl->addMultiCellWidget(m_rulersChBox, 4, 4, 0, 2);
  gl->addMultiCell(new TQSpacerItem(0, 0), 5, 5, 0, 2);

  connect(layoutBtn, TQ_SIGNAL(clicked()), TQ_SLOT(pageLayoutDlg()));
  connect(fontBtn, TQ_SIGNAL(clicked()), TQ_SLOT(fontDlg()));
  connect(m_unitCombo, TQ_SIGNAL(activated(int)), TQ_SLOT(unitChanged(int)));
}

void KivioOptionsDialog::initGrid()
{
  TQFrame* page = addPage(i18n("Grid"), i18n("Grid Settings"), BarIcon( "grid", TDEIcon::SizeMedium ));
  m_gridIndex = pageIndex(page);

  KoUnit::Unit unit = static_cast<KivioView*>(parent())->doc()->unit();
  KivioGridData d = static_cast<KivioView*>(parent())->doc()->grid();
  double pgw = m_layout.ptWidth;
  double pgh = m_layout.ptHeight;
  double fw = Kivio::Config::gridXSpacing();
  double fh = Kivio::Config::gridYSpacing();
  double sw = Kivio::Config::gridXSnap();
  double sh = Kivio::Config::gridYSnap();

  m_gridChBox = new TQCheckBox(i18n("Show &grid"), page);
  m_gridChBox->setChecked(Kivio::Config::showGrid());
  m_snapChBox = new TQCheckBox(i18n("Snap to g&rid"), page);
  m_snapChBox->setChecked(Kivio::Config::snapGrid());
  TQLabel* gridColorLbl = new TQLabel(i18n("Grid &color:"), page);
  m_gridColorBtn = new KColorButton(Kivio::Config::gridColor(), page);
  gridColorLbl->setBuddy(m_gridColorBtn);
  TQGroupBox* spacingGrp = new TQGroupBox(2, TQt::Horizontal, i18n("Spacing"), page);
  TQLabel* spaceHorizLbl = new TQLabel(i18n("&Horizontal:"), spacingGrp);
  m_spaceHorizUSpin = new KoUnitDoubleSpinBox(spacingGrp, 0.0, pgw, 0.1,
    fw, unit);
  spaceHorizLbl->setBuddy(m_spaceHorizUSpin);
  TQLabel* spaceVertLbl = new TQLabel(i18n("&Vertical:"), spacingGrp);
  m_spaceVertUSpin = new KoUnitDoubleSpinBox(spacingGrp, 0.0, pgh, 0.1,
    fh, unit);
  spaceVertLbl->setBuddy(m_spaceVertUSpin);
  TQGroupBox* snapGrp = new TQGroupBox(2, TQt::Horizontal, i18n("Snap Distance"), page);
  TQLabel* snapHorizLbl = new TQLabel(i18n("H&orizontal:"), snapGrp);
  m_snapHorizUSpin = new KoUnitDoubleSpinBox(snapGrp, 0.0, fw, 0.1,
    sw, unit);
  snapHorizLbl->setBuddy(m_snapHorizUSpin);
  TQLabel* snapVertLbl = new TQLabel(i18n("V&ertical:"), snapGrp);
  m_snapVertUSpin = new KoUnitDoubleSpinBox(snapGrp, 0.0, fh, 0.1,
    sh, unit);
  snapVertLbl->setBuddy(m_snapVertUSpin);

  TQGridLayout* gl = new TQGridLayout(page);
  gl->setSpacing(KDialog::spacingHint());
  gl->addMultiCellWidget(m_gridChBox, 0, 0, 0, 2);
  gl->addMultiCellWidget(m_snapChBox, 1, 1, 0, 2);
  gl->addWidget(gridColorLbl, 2, 0);
  gl->addWidget(m_gridColorBtn, 2, 1);
  gl->addItem(new TQSpacerItem(0, 0), 2, 2);
  gl->addMultiCellWidget(spacingGrp, 3, 3, 0, 2);
  gl->addMultiCellWidget(snapGrp, 4, 4, 0, 2);
  gl->addMultiCell(new TQSpacerItem(0, 0), 5, 5, 0, 2);

  connect(m_spaceHorizUSpin, TQ_SIGNAL(valueChanged(double)), TQ_SLOT(setMaxHorizSnap(double)));
  connect(m_spaceVertUSpin, TQ_SIGNAL(valueChanged(double)), TQ_SLOT(setMaxVertSnap(double)));
}

void KivioOptionsDialog::applyPage()
{
  KivioView* view = static_cast<KivioView*>(parent());
  view->doc()->setUnit(static_cast<KoUnit::Unit>(m_unitCombo->currentItem()));
  Kivio::Config::setUnit(KoUnit::unitName(view->doc()->unit()));
  Kivio::Config::setDefaultPageLayout(m_layout);
  Kivio::Config::setFont(m_font);
  view->doc()->setDefaultFont(m_font);
  view->togglePageMargins(m_marginsChBox->isChecked());
  view->toggleShowRulers(m_rulersChBox->isChecked());
}

void KivioOptionsDialog::applyGrid()
{
  KivioGridData d;
  Kivio::Config::setGridXSpacing(m_spaceHorizUSpin->value());
  Kivio::Config::setGridYSpacing(m_spaceVertUSpin->value());
  Kivio::Config::setGridXSnap(m_snapHorizUSpin->value());
  Kivio::Config::setGridYSnap(m_snapVertUSpin->value());
  Kivio::Config::setShowGrid(m_gridChBox->isChecked());
  Kivio::Config::setSnapGrid(m_snapChBox->isChecked());
  Kivio::Config::setGridColor(m_gridColorBtn->color());
  KivioView* view = static_cast<KivioView*>(parent());
  view->doc()->updateView(0);
}

void KivioOptionsDialog::defaultPage()
{
  m_layout = Kivio::Config::defaultPageLayout();
  m_font = Kivio::Config::font();
  m_unitCombo->setCurrentItem(KoUnit::unit(Kivio::Config::unit()));
  unitChanged(m_unitCombo->currentItem());
  setLayoutText(m_layout);
  setFontText(m_font);
  m_marginsChBox->setChecked(true);
  m_rulersChBox->setChecked(true);
}

void KivioOptionsDialog::defaultGrid()
{
  m_spaceHorizUSpin->changeValue(Kivio::Config::gridXSpacing());
  m_spaceVertUSpin->changeValue(Kivio::Config::gridYSpacing());
  m_snapHorizUSpin->changeValue(Kivio::Config::gridXSnap());
  m_snapVertUSpin->changeValue(Kivio::Config::gridYSnap());
  m_gridChBox->setChecked(Kivio::Config::showGrid());
  m_snapChBox->setChecked(Kivio::Config::snapGrid());
  m_gridColorBtn->setColor(Kivio::Config::gridColor());
}

void KivioOptionsDialog::setLayoutText(const KoPageLayout& l)
{
  KoUnit::Unit unit = static_cast<KoUnit::Unit>(m_unitCombo->currentItem());
  TQString txt = i18n("Format: %1, Width: %2 %4, Height: %3 %5").arg(
    KoPageFormat::formatString(l.format)).arg(KoUnit::toUserValue(l.ptWidth, unit))
    .arg(KoUnit::toUserValue(l.ptHeight, unit)).arg(KoUnit::unitName(unit)).arg(
    KoUnit::unitName(unit));
  m_layoutTxtLbl->setText(txt);
}

void KivioOptionsDialog::pageLayoutDlg()
{
  KoHeadFoot headfoot;
  int tabs = FORMAT_AND_BORDERS | DISABLE_UNIT;
  KoUnit::Unit unit = static_cast<KoUnit::Unit>(m_unitCombo->currentItem());

  if(KoPageLayoutDia::pageLayout(m_layout, headfoot, tabs, unit))
  {
    setLayoutText(m_layout);
  }
}

void KivioOptionsDialog::unitChanged(int u)
{
  KoUnit::Unit unit = static_cast<KoUnit::Unit>(u);
  setLayoutText(m_layout);
  m_snapHorizUSpin->setUnit(unit);
  m_snapVertUSpin->setUnit(unit);
  m_spaceHorizUSpin->setUnit(unit);
  m_spaceVertUSpin->setUnit(unit);
}

void KivioOptionsDialog::slotOk()
{
  slotApply();
  accept();
}

void KivioOptionsDialog::slotApply()
{
  applyPage();
  applyGrid();
  Kivio::Config::self()->writeConfig();
}

void KivioOptionsDialog::slotDefault()
{
  bool defaults = Kivio::Config::self()->useDefaults(true);
  defaultPage();
  defaultGrid();
  Kivio::Config::self()->useDefaults(defaults);
}

void KivioOptionsDialog::setMaxHorizSnap(double v)
{
  KoUnit::Unit unit = static_cast<KoUnit::Unit>(m_unitCombo->currentItem());
  m_snapHorizUSpin->setMaxValue(KoUnit::fromUserValue(v, unit));
}

void KivioOptionsDialog::setMaxVertSnap(double v)
{
  KoUnit::Unit unit = static_cast<KoUnit::Unit>(m_unitCombo->currentItem());
  m_snapVertUSpin->setMaxValue(KoUnit::fromUserValue(v, unit));
}

void KivioOptionsDialog::setFontText(const TQFont& f)
{
  TQString txt = f.family() + " " + TQString::number(f.pointSize());
  m_fontTxtLbl->setText(txt);
}

void KivioOptionsDialog::fontDlg()
{
  if(TDEFontDialog::getFont(m_font, false, this) == TQDialog::Accepted) {
    setFontText(m_font);
  }
}

#include "kiviooptionsdialog.moc"
