// This module implements the TQextScintillaLexerBash class.
//
// Copyright (c) 2006
// 	Riverbank Computing Limited <info@riverbankcomputing.co.uk>
// 
// This file is part of TQScintilla.
// 
// This copy of TQScintilla is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option) any
// later version.
// 
// TQScintilla is supplied in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
// FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
// details.
// 
// You should have received a copy of the GNU General Public License along with
// TQScintilla; see the file LICENSE.  If not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <tqcolor.h>
#include <tqfont.h>
#include <tqsettings.h>

#include "tqextscintillalexerbash.h"


// The ctor.
TQextScintillaLexerBash::TQextScintillaLexerBash(TQObject *parent,
					       const char *name)
	: TQextScintillaLexer(parent,name), fold_comments(false),
	  fold_compact(true)
{
}


// The dtor.
TQextScintillaLexerBash::~TQextScintillaLexerBash()
{
}


// Returns the language name.
const char *TQextScintillaLexerBash::language() const
{
	return "Bash";
}


// Returns the lexer name.
const char *TQextScintillaLexerBash::lexer() const
{
	return "bash";
}


// Return the style used for braces.
int TQextScintillaLexerBash::braceStyle() const
{
	return Operator;
}


// Return the string of characters that comprise a word.
const char *TQextScintillaLexerBash::wordCharacters() const
{
	return "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789_$@%&";
}


// Returns the foreground colour of the text for a style.
TQColor TQextScintillaLexerBash::color(int style) const
{
	switch (style)
	{
	case Default:
		return TQColor(0x80,0x80,0x80);

	case Error:
	case Backticks:
		return TQColor(0xff,0xff,0x00);

	case Comment:
		return TQColor(0x00,0x7f,0x00);

	case Number:
		return TQColor(0x00,0x7f,0x7f);

	case Keyword:
		return TQColor(0x00,0x00,0x7f);

	case DoubleQuotedString:
	case SingleQuotedString:
	case SingleQuotedHereDocument:
		return TQColor(0x7f,0x00,0x7f);

	case Operator:
	case Identifier:
	case Scalar:
	case ParameterExpansion:
	case HereDocumentDelimiter:
		return TQColor(0x00,0x00,0x00);
	}

	return TQextScintillaLexer::color(style);
}


// Returns the end-of-line fill for a style.
bool TQextScintillaLexerBash::eolFill(int style) const
{
	switch (style)
	{
	case SingleQuotedHereDocument:
		return true;
	}

	return false;
}


// Returns the font of the text for a style.
TQFont TQextScintillaLexerBash::font(int style) const
{
	TQFont f;

	switch (style)
	{
	case Comment:
#if defined(Q_OS_WIN)
		f = TQFont("Comic Sans MS",9);
#else
		f = TQFont("Bitstream Vera Serif",9);
#endif
		break;

	case Keyword:
	case Operator:
		f = TQextScintillaLexer::font(style);
		f.setBold(true);
		break;

	case DoubleQuotedString:
	case SingleQuotedString:
#if defined(Q_OS_WIN)
		f = TQFont("Courier New",10);
#else
		f = TQFont("Bitstream Vera Sans Mono",9);
#endif
		break;

	default:
		f = TQextScintillaLexer::font(style);
	}

	return f;
}


// Returns the set of keywords.
const char *TQextScintillaLexerBash::keywords(int set) const
{
	if (set == 1)
		return
			"alias ar asa awk banner basename bash bc bdiff break "
			"bunzip2 bzip2 cal calendar case cat cc cd chmod "
			"cksum clear cmp col comm compress continue cp cpio "
			"crypt csplit ctags cut date dc dd declare deroff dev "
			"df diff diff3 dircmp dirname do done du echo ed "
			"egrep elif else env esac eval ex exec exit expand "
			"export expr false fc fgrep fi file find fmt fold for "
			"function functions getconf getopt getopts grep gres "
			"hash head help history iconv id if in integer jobs "
			"join kill local lc let line ln logname look ls m4 "
			"mail mailx make man mkdir more mt mv newgrp nl nm "
			"nohup ntps od pack paste patch pathchk pax pcat perl "
			"pg pr print printf ps pwd read readonly red return "
			"rev rm rmdir sed select set sh shift size sleep sort "
			"spell split start stop strings strip stty sum "
			"suspend sync tail tar tee test then time times touch "
			"tr trap true tsort tty type typeset ulimit umask "
			"unalias uname uncompress unexpand uniq unpack unset "
			"until uudecode uuencode vi vim vpax wait wc whence "
			"which while who wpaste wstart xargs zcat "

			"chgrp chown chroot dir dircolors factor groups "
			"hostid install link md5sum mkfifo mknod nice pinky "
			"printenv ptx readlink seq sha1sum shred stat su tac "
			"unlink users vdir whoami yes";

	return 0;
}


// Returns the user name of a style.
TQString TQextScintillaLexerBash::description(int style) const
{
	switch (style)
	{
	case Default:
		return tr("Default");

	case Error:
		return tr("Error");

	case Comment:
		return tr("Comment");

	case Number:
		return tr("Number");

	case Keyword:
		return tr("Keyword");

	case DoubleQuotedString:
		return tr("Double-quoted string");

	case SingleQuotedString:
		return tr("Single-quoted string");

	case Operator:
		return tr("Operator");

	case Identifier:
		return tr("Identifier");

	case Scalar:
		return tr("Scalar");

	case ParameterExpansion:
		return tr("Parameter expansion");

	case Backticks:
		return tr("Backticks");

	case HereDocumentDelimiter:
		return tr("Here document delimiter");

	case SingleQuotedHereDocument:
		return tr("Single-quoted here document");
	}

	return TQString();
}


// Returns the background colour of the text for a style.
TQColor TQextScintillaLexerBash::paper(int style) const
{
	switch (style)
	{
	case Error:
		return TQColor(0xff,0x00,0x00);

	case Scalar:
		return TQColor(0xff,0xe0,0xe0);

	case ParameterExpansion:
		return TQColor(0xff,0xff,0xe0);

	case Backticks:
		return TQColor(0xa0,0x80,0x80);

	case HereDocumentDelimiter:
	case SingleQuotedHereDocument:
		return TQColor(0xdd,0xd0,0xdd);
	}

	return TQextScintillaLexer::paper(style);
}


// Refresh all properties.
void TQextScintillaLexerBash::refreshProperties()
{
	setCommentProp();
	setCompactProp();
}


// Read properties from the settings.
bool TQextScintillaLexerBash::readProperties(TQSettings &qs,const TQString &prefix)
{
	bool rc = true;
	bool ok, flag;

	// Read the fold comments flag.
	flag = qs.readBoolEntry(prefix + "foldcomments",false,&ok);

	if (ok)
		fold_comments = flag;
	else
		rc = false;

	// Read the fold compact flag.
	flag = qs.readBoolEntry(prefix + "foldcompact",true,&ok);

	if (ok)
		fold_compact = flag;
	else
		rc = false;

	return rc;
}


// Write properties to the settings.
bool TQextScintillaLexerBash::writeProperties(TQSettings &qs,const TQString &prefix) const
{
	bool rc = true;

	// Write the fold comments flag.
	if (!qs.writeEntry(prefix + "foldcomments",fold_comments))
		rc = false;

	// Write the fold compact flag.
	if (!qs.writeEntry(prefix + "foldcompact",fold_compact))
		rc = false;

	return rc;
}


// Return true if comments can be folded.
bool TQextScintillaLexerBash::foldComments() const
{
	return fold_comments;
}


// Set if comments can be folded.
void TQextScintillaLexerBash::setFoldComments(bool fold)
{
	fold_comments = fold;

	setCommentProp();
}


// Set the "fold.comment" property.
void TQextScintillaLexerBash::setCommentProp()
{
	emit propertyChanged("fold.comment",(fold_comments ? "1" : "0"));
}


// Return true if folds are compact.
bool TQextScintillaLexerBash::foldCompact() const
{
	return fold_compact;
}


// Set if folds are compact
void TQextScintillaLexerBash::setFoldCompact(bool fold)
{
	fold_compact = fold;

	setCompactProp();
}


// Set the "fold.compact" property.
void TQextScintillaLexerBash::setCompactProp()
{
	emit propertyChanged("fold.compact",(fold_compact ? "1" : "0"));
}

#include "tqextscintillalexerbash.moc"
