/*  This file is part of the KDE project
    Copyright (C) 2009 Jaroslav Reznik <jreznik@redhat.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.

*/

#include "polkit-listener.h"
#include "AuthDialog.h"

#include <kdebug.h>
#include <tdelocale.h>
#include "twin.h"

#include <PolkitTQt/Agent/Listener>
#include <PolkitTQt/Agent/Session>
#include <PolkitTQt/Subject>
#include <PolkitTQt/Identity>
#include <PolkitTQt/Details>

using namespace PolkitTQt;
using namespace PolkitTQt::Agent;

PolkitListener::PolkitListener(TQObject *parent) : Listener(parent), m_inProgress(false),
        m_selectedUser(0), m_result(nullptr), m_session(nullptr)
{
  PolkitTQt::UnixSessionSubject session(getpid());
  if (!registerListener(session, "/org/trinitydesktop/PolKit1AuthAgent"))
  {
    kdWarning() << "Could not initiate DBus listener!" << endl;
  }
  kdDebug() << "Listener online" << endl;
}

void PolkitListener::clearSession()
{
  if (m_session)
  {
    m_session->deleteLater();
    m_session = nullptr;
  }
}

void PolkitListener::initiateAuthentication(const TQString &actionId, const TQString &message,
        const TQString &iconName, const PolkitTQt::Details &details,
        const TQString &cookie, const Identity::List &identities, AsyncResult* result)
{
  kdDebug() << "Initiating authentication" << endl;

  if (m_inProgress)
  {
    result->setError(i18n("Another client is already authenticating, please try again later."));
    result->setCompleted();
    kdDebug() << "Another client is already authenticating, please try again later." << endl;
    return;
  }

  m_identities = identities;
  m_cookie = cookie;
  m_result = result;
  clearSession();

  m_inProgress = true;

  m_dialog = new AuthDialog(actionId, message, iconName, details, identities);
  connect(m_dialog, TQ_SIGNAL(okClicked()), TQ_SLOT(dialogAccepted()));
  connect(m_dialog, TQ_SIGNAL(cancelClicked()), TQ_SLOT(dialogCanceled()));
  connect(m_dialog, TQ_SIGNAL(adminUserSelected(const PolkitTQt::Identity&)),
          TQ_SLOT(userSelected(const PolkitTQt::Identity&)));

  m_dialog->setOptions();
  m_dialog->show();
	KWin::forceActiveWindow(m_dialog->winId());

  if (identities.count() == 1)
  {
    m_selectedUser = identities[0];
  }
  else
  {
    m_selectedUser = m_dialog->adminUserSelected();
  }

  m_numTries = 0;
  tryAgain();
}

void PolkitListener::tryAgain()
{
  kdDebug() << "Trying again" << endl;
  //  test!!!
  m_wasCancelled = false;

  // We will create new session only when some user is selected
  if (m_selectedUser.isValid())
  {
    m_session = new Session(m_selectedUser, m_cookie, m_result);
    connect(m_session, TQ_SIGNAL(request(const TQString&, bool)), this,
            TQ_SLOT(request(const TQString&, bool)));
    connect(m_session, TQ_SIGNAL(completed(bool)), this, TQ_SLOT(completed(bool)));
    connect(m_session, TQ_SIGNAL(showError(const TQString&)), this,
            TQ_SLOT(showError(const TQString&)));
    m_session->initiate();
  }
}

void PolkitListener::finishObtainPrivilege()
{
  kdDebug() << "Finishing obtaining privileges" << endl;

  // Number of tries increase only when some user is selected
  if (m_selectedUser.isValid())
  {
    m_numTries++;
  }

  if (!m_gainedAuthorization && !m_wasCancelled && m_dialog)
  {
    m_dialog->authenticationFailure();

    if (m_numTries < 3)
    {
      clearSession();
      tryAgain();
      return;
    }
  }

  if (m_session)
  {
    m_session->result()->setCompleted();
  }
  else
  {
    m_result->setCompleted();
  }
  clearSession();

  if (m_dialog)
  {
    m_dialog->hide();
    m_dialog->deleteLater();
    m_dialog = nullptr;
  }

  m_inProgress = false;

  kdDebug() << "Finish obtain authorization:" << m_gainedAuthorization << endl;
}

bool PolkitListener::initiateAuthenticationFinish()
{
  kdDebug() << "Finishing authentication" << endl;
  return true;
}

void PolkitListener::cancelAuthentication()
{
  kdDebug() << "Cancelling authentication" << endl;
  m_wasCancelled = true;
  finishObtainPrivilege();
}

void PolkitListener::request(const TQString &request, bool _)
{
  kdDebug() << "Request: " << request << endl;
  if (m_dialog)
  {
    m_dialog->setRequest(request, m_selectedUser.isValid() &&
            m_selectedUser.toString() == "unix-user:root");
  }
}

void PolkitListener::completed(bool gainedAuthorization)
{
  kdDebug() << "Completed: " << gainedAuthorization << endl;
  m_gainedAuthorization = gainedAuthorization;
  finishObtainPrivilege();
}

void PolkitListener::showError(const TQString &text)
{
  kdDebug() << "Error: " << text << endl;
}

void PolkitListener::dialogAccepted()
{
  kdDebug() << "Dialog accepted" << endl;
  if (m_session)
  {
    m_session->setResponse(m_dialog->password());
  }
}

void PolkitListener::dialogCanceled()
{
  kdDebug() << "Dialog cancelled" << endl;
  m_wasCancelled = true;
  if (m_session)
  {
    m_session->cancel();
  }
  finishObtainPrivilege();
}

void PolkitListener::userSelected(const PolkitTQt::Identity &identity)
{
  m_selectedUser = identity;
  // If some user is selected we must destroy existing session
  clearSession();
  tryAgain();
}

#include "polkit-listener.moc"

