/***************************************************************************
 *   Copyright (C) 2005 by David Saxton                                    *
 *   david@bluehaze.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "itemdocument.h"
#include "itemdocumentdata.h"
#include "drawpart.h"
#include "variant.h"

#include <tdelocale.h>
#include <tqbitarray.h>

DrawPart::DrawPart( ItemDocument *itemDocument, bool newItem, const char *id )
	: Item( itemDocument, newItem, id )
{
	itemDocument->registerItem(this);
}


DrawPart::~DrawPart()
{
}


int DrawPart::rtti() const
{
	return ItemDocument::RTTI::DrawPart;
}


Variant * DrawPart::createProperty( const TQString & id, Variant::Type::Value type )
{
	if ( type == Variant::Type::PenStyle )
	{
		TQStringList penStyles;
		penStyles << DrawPart::penStyleToName(TQt::SolidLine) << DrawPart::penStyleToName(TQt::DashLine)
				<< DrawPart::penStyleToName(TQt::DotLine) << DrawPart::penStyleToName(TQt::DashDotLine)
				<< DrawPart::penStyleToName(TQt::DashDotDotLine);
	
		Variant * v = createProperty( id, Variant::Type::String );
		v->setType( Variant::Type::PenStyle );
		v->setAllowed(penStyles);
		return v;
	}
	
	if ( type == Variant::Type::PenCapStyle )
	{
		TQStringList penCapStyles;
		penCapStyles << DrawPart::penCapStyleToName(TQt::FlatCap) << DrawPart::penCapStyleToName(TQt::SquareCap)
				<< DrawPart::penCapStyleToName(TQt::RoundCap);
		
		Variant * v = createProperty( id, Variant::Type::String );
		v->setType( Variant::Type::PenCapStyle );
		v->setAllowed(penCapStyles);
		return v;
	}
	
	return Item::createProperty( id, type );
}


TQt::PenStyle DrawPart::getDataPenStyle( const TQString & id )
{
	return nameToPenStyle( dataString(id) );
}
TQt::PenCapStyle DrawPart::getDataPenCapStyle( const TQString & id )
{
	return nameToPenCapStyle( dataString(id) );
}
void DrawPart::setDataPenStyle( const TQString & id, TQt::PenStyle value )
{
	property(id)->setValue( penStyleToName(value) );
}
void DrawPart::setDataPenCapStyle( const TQString & id, TQt::PenCapStyle value )
{
	property(id)->setValue( penCapStyleToName(value) );
}


ItemData DrawPart::itemData() const
{
	ItemData itemData = Item::itemData();
	
	const VariantDataMap::const_iterator end = m_variantData.end();
	for ( VariantDataMap::const_iterator it = m_variantData.begin(); it != end; ++it )
	{
		switch( it.data()->type() )
		{
			case Variant::Type::PenStyle:
				itemData.dataString[it.key()] = penStyleToID( nameToPenStyle( it.data()->value().toString() ) );
				break;
			case Variant::Type::PenCapStyle:
				itemData.dataString[it.key()] = penCapStyleToID( nameToPenCapStyle( it.data()->value().toString() ) );
				break;
			case Variant::Type::String:
			case Variant::Type::FileName:
			case Variant::Type::Port:
			case Variant::Type::Pin:
			case Variant::Type::VarName:
			case Variant::Type::Combo:
			case Variant::Type::Select:
			case Variant::Type::Multiline:
			case Variant::Type::Int:
			case Variant::Type::Double:
			case Variant::Type::Color:
			case Variant::Type::Bool:
			case Variant::Type::Raw:
			case Variant::Type::SevenSegment:
			case Variant::Type::KeyPad:
			case Variant::Type::None:
				// All of these are handled by Item
				break;
		}
	}
	
	return itemData;
}


void DrawPart::restoreFromItemData( const ItemData &itemData )
{
	Item::restoreFromItemData(itemData);
	
	const TQStringMap::const_iterator stringEnd = itemData.dataString.end();
	for ( TQStringMap::const_iterator it = itemData.dataString.begin(); it != stringEnd; ++it )
	{
		VariantDataMap::iterator vit = m_variantData.find(it.key());
		if ( vit == m_variantData.end() )
			continue;
		
		if ( vit.data()->type() == Variant::Type::PenStyle )
			setDataPenStyle( it.key(), idToPenStyle( it.data() ) );
		
		else if ( vit.data()->type() == Variant::Type::PenCapStyle )
			setDataPenCapStyle( it.key(), idToPenCapStyle( it.data() ) );
	}
}



TQString DrawPart::penStyleToID( TQt::PenStyle style )
{
	switch (style)
	{
		case TQt::SolidLine:
			return "SolidLine";
		case TQt::NoPen:
			return "NoPen";
		case TQt::DashLine:
			return "DashLine";
		case TQt::DotLine:
			return "DotLine";
		case TQt::DashDotLine:
			return "DashDotLine";
		case TQt::DashDotDotLine:
			return "DashDotDotLine";
		case TQt::MPenStyle:
		default:
			return ""; // ?!
	}
}
TQt::PenStyle DrawPart::idToPenStyle( const TQString & id )
{
	if ( id == "NoPen" )
		return TQt::NoPen;
	if ( id == "DashLine" )
		return TQt::DashLine;
	if ( id == "DotLine" )
		return TQt::DotLine;
	if ( id == "DashDotLine" )
		return TQt::DashDotLine;
	if ( id == "DashDotDotLine" )
		return TQt::DashDotDotLine;
	return TQt::SolidLine;
}
TQString DrawPart::penCapStyleToID( TQt::PenCapStyle style )
{
	switch (style)
	{
		case TQt::FlatCap:
			return "FlatCap";
		case TQt::SquareCap:
			return "SquareCap";
		case TQt::RoundCap:
			return "RoundCap";
		case TQt::MPenCapStyle:
		default:
			return ""; // ?!
	}
}
TQt::PenCapStyle DrawPart::idToPenCapStyle( const TQString & id )
{
	if ( id == "SquareCap" )
		return TQt::SquareCap;
	if ( id == "RoundCap" )
		return TQt::RoundCap;
	return TQt::FlatCap;
}

TQString DrawPart::penStyleToName( TQt::PenStyle style )
{
	switch (style)
	{
		case TQt::SolidLine:
			return i18n("Solid");
		case TQt::NoPen:
			return i18n("None");
		case TQt::DashLine:
			return i18n("Dash");
		case TQt::DotLine:
			return i18n("Dot");
		case TQt::DashDotLine:
			return i18n("Dash Dot");
		case TQt::DashDotDotLine:
			return i18n("Dash Dot Dot");
		case TQt::MPenStyle:
		default:
			return ""; // ?!
	}
}
TQt::PenStyle DrawPart::nameToPenStyle( const TQString & name )
{
	if ( name == i18n("None") )
		return TQt::NoPen;
	if ( name == i18n("Dash") )
		return TQt::DashLine;
	if ( name == i18n("Dot") )
		return TQt::DotLine;
	if ( name == i18n("Dash Dot") )
		return TQt::DashDotLine;
	if ( name == i18n("Dash Dot Dot") )
		return TQt::DashDotDotLine;
	return TQt::SolidLine;
}
TQString DrawPart::penCapStyleToName( TQt::PenCapStyle style )
{
	switch (style)
	{
		case TQt::FlatCap:
			return i18n("Flat");
		case TQt::SquareCap:
			return i18n("Square");
		case TQt::RoundCap:
			return i18n("Round");
		case TQt::MPenCapStyle:
		default:
			return ""; // ?!
	}
}
TQt::PenCapStyle DrawPart::nameToPenCapStyle( const TQString & name )
{
	if ( name == i18n("Square") )
		return TQt::SquareCap;
	if ( name == i18n("Round") )
		return TQt::RoundCap;
	return TQt::FlatCap;
}

