/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Torben Weis <weis@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef __kspread_undo_h__
#define __kspread_undo_h__

#include <tqptrstack.h>
#include <tqstring.h>
#include <tqrect.h>
#include <tqptrlist.h>
#include <tqvaluelist.h>
#include <tqmap.h>

#include <KoUnit.h>
#include <KoPageLayout.h>

#include "kspread_doc.h"
#include "region.h"

namespace KSpread
{
class ColumnFormat;
class Doc;
class Format;
class Region;
class RowFormat;
class Sheet;
class Undo;
class UndoResizeColRow;

struct rowSize {
int rowNumber;
double rowHeight;
};

struct columnSize {
int columnNumber;
double columnWidth;
};

struct textOfCell {
int row;
int col;
TQString text;
};

struct layoutTextCell {
int row;
int col;
Format * l;
TQString text;
};

struct layoutCell {
int row;
int col;
Format *l;
};

struct layoutColumn {
int col;
ColumnFormat *l;
};

struct layoutRow {
int row;
RowFormat *l;
};

struct styleCell {
  int row;
  int col;
  TQString action;
};

class FormulaOfCell
{
public:
    FormulaOfCell(): m_sheetName(0) {}
    FormulaOfCell( TQString & sheetName, int col, int row, TQString & formula )
        : m_sheetName( sheetName ), m_col( col ), m_row( row ), m_formula( formula )
    {}

    TQString sheetName() const { return m_sheetName; }
    TQString formula() const { return m_formula; }
    int col() const { return m_col; }
    int row() const { return m_row; }

private:
    TQString m_sheetName;
    int m_col;
    int m_row;
    TQString m_formula;
};

/**
 * Abstract base class. Every undo/redo action must
 * derive from this class.
 */
class UndoAction
{
public:
    UndoAction( Doc *_doc ) { m_pDoc = _doc; m_pDoc->setModified(true); }
    virtual ~UndoAction() { }

    virtual void undo() = 0;
    virtual void redo() = 0;

    Doc* doc()const { return m_pDoc; }

    TQString getName()const {return name ;}

protected:
    Doc *m_pDoc;
    TQString name;
};

class MacroUndoAction : public UndoAction
{
public:
    MacroUndoAction( Doc * _doc, const TQString & _name );
    virtual ~MacroUndoAction();

    void addCommand(UndoAction *command);

    virtual void undo();
    virtual void redo();

protected:
    TQPtrList<UndoAction> m_commands;
};

class UndoInsertRemoveAction : public UndoAction
{
public:
    UndoInsertRemoveAction( Doc *_doc );
    virtual ~UndoInsertRemoveAction();

    void saveFormulaReference( Sheet *_sheet, int col, int row, TQString & formula );

protected:
    void undoFormulaReference();
    TQValueList<FormulaOfCell> m_lstFormulaCells;
};

class UndoRemoveColumn : public UndoInsertRemoveAction
{
public:
    UndoRemoveColumn( Doc *_doc, Sheet *_sheet, int _column,int _nbCol=0 );
    virtual ~UndoRemoveColumn();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    TQCString m_data;
    int m_iColumn;
    int m_iNbCol;
    TQRect m_printRange;
    TQPair<int, int> m_printRepeatColumns;
};

class UndoInsertColumn : public UndoInsertRemoveAction
{
public:
    UndoInsertColumn( Doc *_doc, Sheet *_sheet, int _column,int _nbCol=0 );
    virtual ~UndoInsertColumn();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    int m_iColumn;
    int m_iNbCol;
};

class UndoRemoveRow : public UndoInsertRemoveAction
{
public:
    UndoRemoveRow( Doc *_doc, Sheet *_sheet, int _row,int _nbRow=0 );
    virtual ~UndoRemoveRow();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    TQCString m_data;
    int m_iRow;
    int m_iNbRow;
    TQRect m_printRange;
    TQPair<int, int> m_printRepeatRows;
};

class UndoInsertRow : public UndoInsertRemoveAction
{
public:
    UndoInsertRow( Doc *_doc, Sheet *_sheet, int _row,int _nbRow=0 );
    virtual ~UndoInsertRow();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    int m_iRow;
    int m_iNbRow;
};


class UndoHideColumn : public UndoAction
{
public:
    UndoHideColumn( Doc *_doc, Sheet *_sheet, int _column,int _nbCol=0, TQValueList<int>listCol=TQValueList<int>() );
    virtual ~UndoHideColumn();

    virtual void undo();
    virtual void redo();
    void createList( TQValueList<int>&list,Sheet *_tab );

protected:
    TQString m_sheetName;
    int m_iColumn;
    int m_iNbCol;
    TQValueList<int> listCol;
};

class UndoHideRow : public UndoAction
{
public:
    UndoHideRow( Doc *_doc, Sheet *_sheet, int _column,int _nbCol=0, TQValueList<int>_listRow=TQValueList<int>() );
    virtual ~UndoHideRow();

    virtual void undo();
    virtual void redo();
protected:
    void createList( TQValueList<int>&list,Sheet *_tab );

    TQString m_sheetName;
    int m_iRow;
    int m_iNbRow;
    TQValueList<int> listRow;
};

class UndoShowColumn : public UndoAction
{
public:
    UndoShowColumn( Doc *_doc, Sheet *_sheet, int _column,int _nbCol=0, TQValueList<int>_list=TQValueList<int>() );
    virtual ~UndoShowColumn();

    virtual void undo();
    virtual void redo();
protected:
    void createList( TQValueList<int>&list,Sheet *_tab );

    TQString m_sheetName;
    int m_iColumn;
    int m_iNbCol;
    TQValueList<int> listCol;
};

class UndoShowRow : public UndoAction
{
public:
    UndoShowRow( Doc *_doc, Sheet *_sheet, int _column,int _nbCol=0, TQValueList<int>list=TQValueList<int>() );
    virtual ~UndoShowRow();

    virtual void undo();
    virtual void redo();

protected:
    void createList( TQValueList<int>&list,Sheet *_tab );
    TQString m_sheetName;
    int m_iRow;
    int m_iNbRow;
    TQValueList<int> listRow;
};


class UndoPaperLayout : public UndoAction
{
public:
    UndoPaperLayout( Doc *_doc, Sheet *_sheet );
    virtual ~UndoPaperLayout();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    KoPageLayout m_pl;
    KoPageLayout m_plRedo;
    KoHeadFoot m_hf;
    KoHeadFoot m_hfRedo;
    KoUnit::Unit m_unit;
    KoUnit::Unit m_unitRedo;
    bool m_printGrid;
    bool m_printGridRedo;
    bool m_printCommentIndicator;
    bool m_printCommentIndicatorRedo;
    bool m_printFormulaIndicator;
    bool m_printFormulaIndicatorRedo;
    TQRect m_printRange;
    TQRect m_printRangeRedo;
    TQPair<int, int> m_printRepeatColumns;
    TQPair<int, int> m_printRepeatColumnsRedo;
    TQPair<int, int> m_printRepeatRows;
    TQPair<int, int> m_printRepeatRowsRedo;
    double m_dZoom;
    double m_dZoomRedo;
    int m_iPageLimitX;
    int m_iPageLimitXRedo;
    int m_iPageLimitY;
    int m_iPageLimitYRedo;
};


class UndoSetText : public UndoAction
{
public:
    UndoSetText( Doc *_doc, Sheet *_sheet, const TQString& _text, int _column, int _row, FormatType _formatType );
    virtual ~UndoSetText();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    int m_iRow;
    int m_iColumn;
    TQString m_strText;
    TQString m_strRedoText;
    FormatType m_eFormatType;
    FormatType m_eFormatTypeRedo;
};

class UndoCellFormat : public UndoAction
{
public:
    UndoCellFormat( Doc *_doc, Sheet *_sheet, const Region &_selection, const TQString &_title );
    virtual ~UndoCellFormat();

    virtual void undo();
    virtual void redo();

protected:
    void copyFormat( TQValueList<layoutCell> &list,TQValueList<layoutColumn> &listCol,TQValueList<layoutRow> &listRow, Sheet* sheet );

    Region m_region;
    TQValueList<layoutCell> m_lstFormats;
    TQValueList<layoutCell> m_lstRedoFormats;
    TQValueList<layoutColumn> m_lstColFormats;
    TQValueList<layoutColumn> m_lstRedoColFormats;
    TQValueList<layoutRow> m_lstRowFormats;
    TQValueList<layoutRow> m_lstRedoRowFormats;

    TQString m_sheetName;
};

class UndoChangeAngle : public UndoAction
{
public:
    UndoChangeAngle( Doc *_doc, Sheet *_sheet, const Region &_selection );
    virtual ~UndoChangeAngle();

    virtual void undo();
    virtual void redo();

protected:

   UndoCellFormat* m_layoutUndo;
   UndoResizeColRow* m_resizeUndo;

};

class UndoDelete : public UndoAction
{
public:
    UndoDelete(Doc *_doc, Sheet *_sheet, const Region& region);
    virtual ~UndoDelete();

    virtual void undo();
    virtual void redo();

protected:
    void createListCell( TQCString &listCell,TQValueList<columnSize> &listCol,TQValueList<rowSize> &listRow, Sheet* sheet );

    Region m_region;
    TQCString m_data;
    TQCString m_dataRedo;
    TQValueList<columnSize> m_lstColumn;
    TQValueList<columnSize> m_lstRedoColumn;
    TQValueList<rowSize> m_lstRow;
    TQValueList<rowSize> m_lstRedoRow;
    TQString m_sheetName;
};

class UndoDragDrop : public UndoAction
{
public:
    UndoDragDrop( Doc * _doc, Sheet * _sheet, const Region& _source, const Region& _target );
    virtual ~UndoDragDrop();

    virtual void undo();
    virtual void redo();

protected:
    Region   m_selectionSource;
    Region   m_selectionTarget;
    TQCString m_dataSource;
    TQCString m_dataTarget;
    TQCString m_dataRedoSource;
    TQCString m_dataRedoTarget;
    TQString  m_sheetName;

    void saveCellRect( TQCString & cells, Sheet * sheet,
                       const Region& region );
};

class UndoResizeColRow : public UndoAction
{
public:
    UndoResizeColRow( Doc *_doc, Sheet *_sheet, const Region &_selection );
    virtual ~UndoResizeColRow();

    virtual void undo();
    virtual void redo();

protected:
    void createList( TQValueList<columnSize> &listCol,TQValueList<rowSize> &listRow, Sheet* sheet );

    Region m_region;
    TQValueList<columnSize> m_lstColumn;
    TQValueList<columnSize> m_lstRedoColumn;
    TQValueList<rowSize> m_lstRow;
    TQValueList<rowSize> m_lstRedoRow;
    TQString m_sheetName;
};

class UndoChangeAreaTextCell : public UndoAction
{
public:
    UndoChangeAreaTextCell( Doc *_doc, Sheet *_sheet, const Region &_selection );
    virtual ~UndoChangeAreaTextCell();

    virtual void undo();
    virtual void redo();

protected:
    void createList( TQMap<TQPoint,TQString> &list, Sheet* sheet );

    Region m_region;
    TQMap<TQPoint,TQString> m_lstTextCell;
    TQMap<TQPoint,TQString> m_lstRedoTextCell;
    TQString m_sheetName;
};

class UndoSort : public UndoAction
{
public:
    UndoSort( Doc *_doc, Sheet *_sheet, const TQRect &_selection);
    virtual ~UndoSort();

    virtual void undo();
    virtual void redo();

protected:
    void copyAll( TQValueList<layoutTextCell> & list, TQValueList<layoutColumn> & listCol,
                  TQValueList<layoutRow> & listRow, Sheet * sheet );

    TQRect m_rctRect;
    TQValueList<layoutTextCell> m_lstFormats;
    TQValueList<layoutTextCell> m_lstRedoFormats;
    TQValueList<layoutColumn> m_lstColFormats;
    TQValueList<layoutColumn> m_lstRedoColFormats;
    TQValueList<layoutRow> m_lstRowFormats;
    TQValueList<layoutRow> m_lstRedoRowFormats;

    TQString m_sheetName;
};

class UndoMergedCell : public UndoAction
{
public:
    UndoMergedCell( Doc *_doc, Sheet *_sheet, int _column, int _row, int _extraX,int _extraY);
    virtual ~UndoMergedCell();

    virtual void undo();
    virtual void redo();

protected:
    int m_iRow;
    int m_iCol;
    int m_iExtraX;
    int m_iExtraY;
    int m_iExtraRedoX;
    int m_iExtraRedoY;
    TQString m_sheetName;
};


class UndoAutofill : public UndoAction
{
public:
    UndoAutofill( Doc *_doc, Sheet *_sheet, const TQRect &_rect );
    virtual ~UndoAutofill();

    virtual void undo();
    virtual void redo();
protected:
    void createListCell( TQCString &list, Sheet* sheet );
    TQRect m_selection;
    TQCString m_data;
    TQCString m_dataRedo;
    TQString m_sheetName;
};

class UndoInsertCellCol : public UndoInsertRemoveAction
{
public:
    UndoInsertCellCol( Doc *_doc, Sheet *_sheet, const TQRect &_rect );
    virtual ~UndoInsertCellCol();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    TQRect m_rect;
};

class UndoInsertCellRow : public UndoInsertRemoveAction
{
public:
    UndoInsertCellRow( Doc *_doc, Sheet *_sheet,const TQRect &_rect );
    virtual ~UndoInsertCellRow();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    TQRect m_rect;
};

class UndoRemoveCellCol : public UndoInsertRemoveAction
{
public:
    UndoRemoveCellCol( Doc *_doc, Sheet *_sheet, const TQRect &_rect );
    virtual ~UndoRemoveCellCol();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    TQRect m_rect;
    TQCString m_data;
};

class UndoRemoveCellRow : public UndoInsertRemoveAction
{
public:
    UndoRemoveCellRow( Doc *_doc, Sheet *_sheet, const TQRect &_rect );
    virtual ~UndoRemoveCellRow();

    virtual void undo();
    virtual void redo();

protected:
    TQString m_sheetName;
    TQRect m_rect;
    TQCString m_data;
};

class UndoConditional : public UndoAction
{
public:
    UndoConditional( Doc *_doc, Sheet *_sheet, const Region & _selection );
    virtual ~UndoConditional();

    virtual void undo();
    virtual void redo();
protected:
    void createListCell( TQCString &list, Sheet* sheet );
    Region m_region;
    TQCString m_data;
    TQCString m_dataRedo;
    TQString m_sheetName;
};

class UndoCellPaste : public UndoAction
{
public:
    UndoCellPaste(Doc *_doc, Sheet *_sheet,
                  int _xshift, int _yshift,
                  const Region& _selection, bool insert, int insertTo = 0);
    virtual ~UndoCellPaste();

    virtual void undo();
    virtual void redo();

protected:
    void createListCell( TQCString &listCell,TQValueList<columnSize> &listCol,TQValueList<rowSize> &listRow, Sheet* sheet );

    Region m_region;
    TQCString m_data;
    TQCString m_dataRedo;
    TQValueList<columnSize> m_lstColumn;
    TQValueList<columnSize> m_lstRedoColumn;
    TQValueList<rowSize> m_lstRow;
    TQValueList<rowSize> m_lstRedoRow;
    int xshift;
    int yshift;
    bool  b_insert;
    int m_iInsertTo;
    TQString m_sheetName;
};


class UndoStyleCell : public UndoAction
{
public:
    UndoStyleCell( Doc *_doc, Sheet *_sheet, const TQRect &_rect );
    virtual ~UndoStyleCell();

    virtual void undo();
    virtual void redo();

protected:
    void createListCell( TQValueList<styleCell> &listCell, Sheet* sheet );
    TQRect m_selection;
    TQValueList<styleCell> m_lstStyleCell;
    TQValueList<styleCell> m_lstRedoStyleCell;
    TQString m_sheetName;
};

class UndoInsertData : public UndoChangeAreaTextCell
{
 public:
    UndoInsertData( Doc * _doc, Sheet * _sheet, TQRect & _selection );
};


class Undo
{
public:
    Undo( Doc *_doc );
    ~Undo();

    void undo();
    void redo();
    void clear();

    void lock();
    void unlock();
    bool isLocked() const ;

    bool hasUndoActions()const { return !m_stckUndo.isEmpty(); }
    bool hasRedoActions()const { return !m_stckRedo.isEmpty(); }

    void appendUndo( UndoAction *_action );

    TQString getUndoName();
    TQString getRedoName();

protected:
    TQPtrStack<UndoAction> m_stckUndo;
    TQPtrStack<UndoAction> m_stckRedo;

    Doc *m_pDoc;
};

} // namespace KSpread

#endif
