/* ============================================================
 * Author: Tom Albers <tomalbers@kde.nl>
 * Date  : 2005-01-01
 * Description : 
 * 
 * Copyright 2005 by Tom Albers
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

/** @file squeezedcombobox.cpp */

// TQt includes.

#include <tqlistbox.h>
#include <tqcombobox.h>
#include <tqpair.h>
#include <tqtimer.h>
#include <tqvaluelist.h>
#include <tqstyle.h>
#include <tqapplication.h>
#include <tqtooltip.h>

// Local includes.

#include "squeezedcombobox.h"

SqueezedComboBoxTip::SqueezedComboBoxTip( TQWidget * parent, SqueezedComboBox* name )
    : TQToolTip( parent )
{
    m_originalWidget = name;
}

void SqueezedComboBoxTip::maybeTip( const TQPoint &pos )
{
    TQListBox* listBox = m_originalWidget->listBox();
    if (!listBox)
        return;

    TQListBoxItem* selectedItem = listBox->itemAt( pos );
    if (selectedItem)
    {
        TQRect positionToolTip = listBox->itemRect( selectedItem );
        TQString toolTipText = m_originalWidget->itemHighlighted();
        if (!toolTipText.isNull())
            tip(positionToolTip, toolTipText);
    }
}

SqueezedComboBox::SqueezedComboBox( TQWidget *parent, const char *name )
    : TQComboBox( parent, name )
{
    setMinimumWidth(100);
    m_timer = new TQTimer(this);
    m_tooltip = new SqueezedComboBoxTip( listBox()->viewport(), this );

    connect(m_timer, TQ_SIGNAL(timeout()),
            TQ_SLOT(slotTimeOut()));
    connect(this, TQ_SIGNAL(activated( int )),
            TQ_SLOT(slotUpdateToolTip( int )));
}

SqueezedComboBox::~SqueezedComboBox()
{
    delete m_tooltip;
    delete m_timer;
}

bool SqueezedComboBox::contains( const TQString& _text ) const
{
    if ( _text.isEmpty() )
        return false;

    const int itemCount = count();
    for (int i = 0; i < itemCount; ++i )
    {
        if ( text(i) == _text )
            return true;
    }
    return false;
}

TQSize SqueezedComboBox::sizeHint() const
{
    constPolish();
    TQFontMetrics fm = fontMetrics();

    int maxW = count() ? 18 : 7 * fm.width(TQChar('x')) + 18;
    int maxH = TQMAX( fm.lineSpacing(), 14 ) + 2;

    return style().sizeFromContents(TQStyle::CT_ComboBox, this,
    TQSize(maxW, maxH)).
            expandedTo(TQApplication::globalStrut());
}

void SqueezedComboBox::insertSqueezedItem(const TQString& newItem, int index)
{
    m_originalItems[index] = newItem;
    insertItem( squeezeText(newItem), index );

    // if this is the first item, set the tooltip.
    if (index == 0)
        slotUpdateToolTip(0);
}

void SqueezedComboBox::resizeEvent ( TQResizeEvent * )
{
    m_timer->start(200, true);
}

void SqueezedComboBox::slotTimeOut()
{
    TQMapIterator<int,TQString> it;
    for (it = m_originalItems.begin() ; it != m_originalItems.end();
         ++it)
    {
        changeItem( squeezeText( it.data() ), it.key() );
    }
}

TQString SqueezedComboBox::squeezeText( const TQString& original)
{
    // not the complete widgetSize is usable. Need to compensate for that.
    int widgetSize = width()-30;
    TQFontMetrics fm( fontMetrics() );

    // If we can fit the full text, return that.
    if (fm.width(original) < widgetSize)
        return(original);

    // We need to squeeze.
    TQString sqItem = original; // prevent empty return value;
    widgetSize = widgetSize-fm.width("...");
    for (uint i = 0 ; i != original.length(); ++i)
    {
        if ( (int)fm.width(original.right(i)) > widgetSize)
        {
            sqItem = TQString("..." + original.right(--i));
            break;
        }
    }
    return sqItem;
}

void SqueezedComboBox::slotUpdateToolTip( int index )
{
    TQToolTip::remove(this);
    TQToolTip::add(this, m_originalItems[index]);
}

TQString SqueezedComboBox::itemHighlighted()
{
    int curItem = this->listBox()->currentItem();
    return m_originalItems[curItem];
}

#include "squeezedcombobox.moc"
