/* ============================================================
 * File  : calselect.cpp
 * Author: Renchi Raju <renchi@pooh.tam.uiuc.edu>
 * Date  : 2003-11-03
 * Description :
 *
 * Copyright 2003 by Renchi Raju <renchi@pooh.tam.uiuc.edu>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqhgroupbox.h>
#include <tqlayout.h>
#include <tqspinbox.h>
#include <tqdatetime.h>
#include <tqlabel.h>
#include <tqframe.h>
#include <tqpixmap.h>

// KDE includes.

#include <tdeglobal.h>
#include <tdelocale.h>
#include <kcalendarsystem.h>
#include <kstandarddirs.h>

// Local includes.

#include "calselect.h"
#include "calsettings.h"
#include "monthwidget.h"

#define MAX_MONTHS (13)

namespace KIPICalendarPlugin
{

CalSelect::CalSelect(KIPI::Interface* interface, TQWidget *parent, const char* name)
         : TQWidget(parent, name)
{
    mwVector_ = new TQPtrVector<MonthWidget>(MAX_MONTHS);
    monthBoxLayout_ = NULL;
    setupView( interface );
}

CalSelect::~CalSelect()
{
    delete mwVector_;
}

void CalSelect::setupView( KIPI::Interface* interface )
{
    TQVBoxLayout *mainLayout = new TQVBoxLayout(this, 6, 11);

    // ----------------------------------------------------------------

    setCaption(i18n("Create Calendar"));
    TQHGroupBox *yearBox = new TQHGroupBox(i18n("Select Year"), this);
    yearBox->layout()->addItem(new TQSpacerItem(5,5,
                                               TQSizePolicy::Expanding,
                                               TQSizePolicy::Minimum));
    yearSpin_ = new TQSpinBox(TDEGlobal::locale()->calendar()->minValidYear(),
                             TDEGlobal::locale()->calendar()->maxValidYear(),
                             1,yearBox);
    yearSpin_->setValue(TDEGlobal::locale()->calendar()->year(TQDate::currentDate()));
    slotYearChanged(yearSpin_->value());

    connect(yearSpin_, TQ_SIGNAL(valueChanged(int)),
            TQ_SLOT(slotYearChanged(int)));

    mainLayout->addWidget(yearBox);

    // ----------------------------------------------------------------

    TQGroupBox *monthBox = new TQGroupBox(i18n("Select Images"), this);
    monthBox->setColumnLayout(0, TQt::Vertical );
    monthBox->layout()->setSpacing( 6 );
    monthBox->layout()->setMargin( 11 );

    monthBoxLayout_ = new TQGridLayout(monthBox->layout());
    monthBoxLayout_->setAlignment( TQt::AlignCenter );

    KURL::List urlList;
    KIPI::ImageCollection images = interface->currentSelection();
    if ( images.isValid() && !images.images().isEmpty())
        urlList = images.images();

    TQDate d;
    TDEGlobal::locale()->calendar()->setYMD(d, yearSpin_->value(), 1, 1);
    unsigned int months = TDEGlobal::locale()->calendar()->monthsInYear(d);
    // span the monthWidgets over 2 rows. inRow should usually be 6 or 7 (for 12 or 13 months)
    int inRow = (months / 2) + ((months % 2) != 0);
    MonthWidget *w;

    for (unsigned int i=0; i<MAX_MONTHS; i++) {
        w = new MonthWidget( interface, monthBox, i+1);
        if (i < urlList.count())
            w->setImage(urlList[i]);
        if (i<months)
            monthBoxLayout_->addWidget(w, i / inRow, i % inRow);
        else
            w->hide();
        mwVector_->insert(i, w);
    }

    TQLabel* tLabel =
        new TQLabel(i18n("Left click on Months to Select Images. "
                        "Right click to Clear Month.\n"
                        "You can also drag and drop images onto the Months"),
                                monthBox);

    monthBoxLayout_->addMultiCellWidget(tLabel, 2, 2, 0, 5);

    mainLayout->addWidget(monthBox);

    // ----------------------------------------------------------------

    mainLayout->addItem(new TQSpacerItem(5,5,TQSizePolicy::Minimum,
                                        TQSizePolicy::Expanding));
}

void CalSelect::slotYearChanged(int year)
{
    int i, months;
    TQDate d, oldD;
    TDEGlobal::locale()->calendar()->setYMD(d, year, 1, 1);
    TDEGlobal::locale()->calendar()->setYMD(oldD, CalSettings::instance()->getYear(), 1, 1);
    months = TDEGlobal::locale()->calendar()->monthsInYear(d);

    if ((TDEGlobal::locale()->calendar()->monthsInYear(oldD) != months) && !mwVector_->isEmpty())
    {
        // hide the last months that are not present on the current year
        for (i=months; (i<TDEGlobal::locale()->calendar()->monthsInYear(oldD)) && (i<(int)mwVector_->count()); i++)
            mwVector_->at(i)->hide();

        // span the monthWidgets over 2 rows. inRow should usually be 6 or 7 (for 12 or 13 months)
        int inRow = (months / 2) + ((months % 2) != 0);
        // remove all the monthWidgets, then readd the needed ones
        for (i=0; i<TDEGlobal::locale()->calendar()->monthsInYear(oldD); i++) {
            monthBoxLayout_->remove(mwVector_->at(i));
        }
        for (i=0; (i<months) && (i<(int)mwVector_->count()); i++) {
            monthBoxLayout_->addWidget(mwVector_->at(i), i / inRow, i % inRow);
            if (mwVector_->at(i)->isHidden())
                mwVector_->at(i)->show();
            mwVector_->at(i)->update();
        }
    }
    CalSettings::instance()->setYear(year);
}

}  // NameSpace KIPICalendarPlugin

#include "calselect.moc"
