/* ============================================================
 *
 * This file is a part of kipi-plugins project
 * http://www.kipi-plugins.org
 *
 * Date        : 2006-05-16
 * Description : a tool to export GPS data to KML file.
 *
 * Copyright (C) 2006-2007 by Stephane Pontier <shadow dot walker at free dot fr>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#ifndef KIPIKMLEXPORTKMLEXPORT_H
#define KIPIKMLEXPORTKMLEXPORT_H

// TQt includes.

#include <tqcolor.h>
#include <tqdir.h>
#include <tqdom.h>

// Local includes.

#include "kmlgpsdataparser.h"

class TQImage;

namespace KIPI 
{
    class BatchProgressDialog;
    class Interface;
}

namespace KIPIGPSSyncPlugin 
{

/**
Exporter to KML

	@author KIPI dev. team
*/
class kmlExport
{

public:

    kmlExport(KIPI::Interface* interface);
    ~kmlExport();

    bool createDir(TQDir dir);

    /*! generate the kml element for pictures with tumbnails
     *  @param interface the kipi interface
     *  @param KURL the URL of the picture
     *  @param kmlAlbum the album used
     */
    void generateImagesthumb(KIPI::Interface* interface, const KURL&, TQDomElement &kmlAlbum);

    /*! Produce a web-friendly file name
     *  otherwise, while google earth works fine, maps.google.com may not find pictures and thumbnail
     *  thank htmlexport
     *  @param the filename
     *  @return the webifyed filename
     */
    TQString webifyFileName(const TQString &fileName);

    /*! Generate a square thumbnail from @fullImage of @size x @size pixels
     *  @param fullImage the original image
     *  @param size the size of the thumbnail
     *  @return the thumbnail
     */
    TQImage generateSquareThumbnail(const TQImage& fullImage, int size);

    /*! Generate a square thumbnail from @fullImage of @size x @size pixels
     *  with a white border
     *  @param fullImage the original image
     *  @param size the size of the thumbnail
     *  @return the thumbnail
     */
    TQImage generateBorderedThumbnail(const TQImage& fullImage, int size);
    void   addTrack(TQDomElement &kmlAlbum);
    void   generate();
    int    getConfig();

public:

    bool             m_localTarget;
    bool             m_optimize_googlemap;
    bool             m_GPXtracks;

    int              m_iconSize;
    int              m_googlemapSize;
    int              m_size;
    int              m_altitudeMode;
    int              m_TimeZone;
    int              m_LineWidth;
    int              m_GPXOpacity;
    int              m_GPXAltitudeMode;

    /** directory used in kmldocument structure */
    TQString          m_imageDir;
    TQString          m_GPXFile;
    TQString          m_UrlDestDir;

    /** temporary directory where everything will be created */
    TQString          m_tempDestDir;

    /** directory selected by user*/
    TQString          m_baseDestDir;

    TQString          m_imgdir;
    TQString          m_KMLFileName;

    TQColor           m_GPXColor;

    KIPI::Interface *m_interface;

private:

    /*! the root document, used to create all TQDomElements */
    TQDomDocument              *kmlDocument;

    /*! the GPS parsed data */
    KMLGPSDataParser           m_gpxParser;

    KIPI::BatchProgressDialog *m_progressDialog;

private:

    void logInfo(const TQString& msg);
    void logError(const TQString& msg);
    void logWarning(const TQString& msg);

    /*!
     *  \fn KIPIKMLExport::kmlExport::addKmlElement(TQDomElement target, TQString tag)
     *  Add a new element
     *  @param target the parent element to which add the element
     *  @param tag the new element name
     *  @return the New element
     */
    TQDomElement addKmlElement(TQDomElement &target, TQString tag)
    {
        TQDomElement kmlElement = kmlDocument->createElement( tag );
        target.appendChild( kmlElement );
        return kmlElement;
    }

    /*!
     *  \fn KIPIKMLExport::kmlExport::addKmlTextElement(TQDomElement target, TQString tag, TQString text)
     *  Add a new element with a text
     *  @param target the parent element to which add the element
     *  @param tag the new element name
     *  @param text the text content of the new element
     *  @return the New element
     */
    TQDomElement addKmlTextElement(TQDomElement &target, TQString tag, TQString text)
    {
        TQDomElement kmlElement  = kmlDocument->createElement( tag );
        target.appendChild( kmlElement );
        TQDomText kmlTextElement = kmlDocument->createTextNode( text );
        kmlElement.appendChild( kmlTextElement );
        return kmlElement;
    }

    /*!
     *  \fn KIPIKMLExport::kmlExport::addKmlHtmlElement(TQDomElement target, TQString tag, TQString text)
     *  Add a new element with html content (html entities are escaped and text is wrapped in a CDATA section)
     *  @param target the parent element to which add the element
     *  @param tag the new element name
     *  @param text the HTML content of the new element
     *  @return the New element
     */
    TQDomElement addKmlHtmlElement(TQDomElement &target, TQString tag, TQString text)
    {
        TQDomElement kmlElement  = kmlDocument->createElement( tag );
        target.appendChild( kmlElement );
        TQDomText kmlTextElement = kmlDocument->createCDATASection( text );
        kmlElement.appendChild( kmlTextElement );
        return kmlElement;
    }
};

} // namespace KIPIGPSSyncPlugin

#endif // KIPIKMLEXPORTKMLEXPORT_H
