/*
 * This file is part of the PolKit-tqt project
 * Copyright (C) 2009 Jaroslav Reznik <jreznik@redhat.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB. If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#ifndef POLKIT_TQT_SUBJECT_H
#define POLKIT_TQT_SUBJECT_H

#include "polkit-tqt-export.h"

typedef struct _PolkitSubject       PolkitSubject;
typedef struct _PolkitUnixProcess   PolkitUnixProcess;
typedef struct _PolkitSystemBusName PolkitSystemBusName;

class TQString;


/**
 * \namespace PolkitTQt PolkitTQt
 *
 * \brief Namespace wrapping PolicyKit-TQt classes
 *
 * This namespace wraps all PolicyKit-TQt classes.
 */
namespace PolkitTQt
{

/**
 * \class Subject polkit-tqt-subject.h Subject
 * \author Jaroslav Reznik <jreznik@redhat.com>
 *
 * \brief This class represents PolicyKit subjects
 *
 * This class encapsulates the PolkitSubject interface.
 *
 * \see UnixProcess
 * \see SystemBusName
 * \see UnixSession
 */
class POLKIT_TQT_EXPORT Subject
{
  public:
    Subject();
    Subject(const Subject &other);
    ~Subject();

    /**
     * Creates a Subject object from PolkitSubject
     *
     * \warning It shouldn't be used directly unless you are completely aware of what are you doing
     *
     * \param pkSubject PolkitSubject object
     */
    explicit Subject(PolkitSubject *pkSubject);

    Subject& operator=(const Subject &other);

    /**
     * Check whether a subject is valid
     *
     * \return true is the subject is valid
     */
    bool isValid() const;

    /**
     * Serialization of object to the string
     *
     * \return Serialized Subject object
     */
    TQString toString() const;

    /**
     * Creates the Subject object from string representation
     *
     * \param string string representation of the object
     *
     * \return the new Subject instance
     */
    static Subject fromString(const TQString &string);

    /**
     * Gets PolkitSubject object.
     *
     * \warning It shouldn't be used directly unless you are completely aware of what are you doing
     *
     * \return Pointer to PolkitSubject instance
     */
    PolkitSubject* subject() const;

    void setSubject(PolkitSubject *subject);

  private:
    class Data;
    Data *d;
};

/**
 * \class UnixProcessSubject polkit-tqt-subject.h Subject
 * \author Jaroslav Reznik <jreznik@redhat.com>
 *
 * \brief A class for representing a UNIX process.
 *
 * To uniquely identify processes, both the process
 * id and the start time of the process (a monotonic
 * increasing value representing the time since the
 * kernel was started) is used.
 *
 * \sa Subject
 */
class POLKIT_TQT_EXPORT UnixProcessSubject : public Subject
{
  public:
    /**
    * Subject constructor, takes one parameter - PID. The start time
    * of process will be looked automatically.
    *
    * \param pid An Unix process PID.
    */
    explicit UnixProcessSubject(TQ_LONG pid);

    /**
    * Subject constructor, takes two parameters - PID and start time.
    *
    * \param pid An Unix process PID.
    * \param startTime An Unix process start time.
    */
    UnixProcessSubject(TQ_LONG pid, TQ_ULONG startTime);

    /**
     * Subject constructor, it creates UnixProcess object from PolkitUnixProcess object
     *
     * \warning Use this only if you are completely aware of what are you doing!
     *
     * \param process PolkitUnixProcess object
     */
    explicit UnixProcessSubject(PolkitUnixProcess *process);

    /**
    * Returns Unix process PID.
    *
    * \return A PID of associated Unix process.
    */
    TQ_LONG pid() const;

    /**
    * Returns Unix process start time.
    *
    * \return A start time of associated Unix process.
    */
    TQ_LONG startTime() const;

    /**
    * Sets Unix process PID.
    *
    * \param pid An Unix process PID.
    */
    void setPid(TQ_LONG pid);
};

/**
 * \class SystemBusNameSubject polkit-tqt-subject.h Subject
 * \author Jaroslav Reznik <jreznik@redhat.com>
 *
 * \brief A class for representing a process owning a unique name on the system bus.
 *
 * \sa Subject
 */
class POLKIT_TQT_EXPORT SystemBusNameSubject : public Subject
{
  public:
    /**
    * Subject constructor, takes one parameter - system bus name.
    *
    * \param name A unique system bus name.
    */
    explicit SystemBusNameSubject(const TQString &name);

    /**
     * Subject constructor, it creates SystemBusName object from PolkitSystemBusName object
     *
     * \warning Use this only if you are completely aware of what are you doing!
     *
     * \param pkSystemBusName PolkitSystemBusName object
     */
    explicit SystemBusNameSubject(PolkitSystemBusName *pkSystemBusName);

    /**
    * Returns system bus name.
    *
    * \return A unique system bus name.
    */
    TQString name() const;

    /**
    * Sets system bus name.
    *
    * \param name System bus name.
    */
    void setName(const TQString &name);
};

/**
 * \class UnixSessionSubject polkit-tqt-subject.h Subject
 * \author Jaroslav Reznik <jreznik@redhat.com>
 *
 * \brief A class for representing unix session.
 *
 * \sa Subject
 */
class POLKIT_TQT_EXPORT UnixSessionSubject : public Subject
{
  public:
    /**
    * Subject constructor, takes one parameter - session id.
    *
    * \param sessionId The session id.
    */
    explicit UnixSessionSubject(const TQString &sessionId);

    /**
    * Subject constructor, takes one parameter - pid of process.
    *
    * Synchronous!
    *
    * \param pid The session's process pid.
    */
    explicit UnixSessionSubject(TQ_LONG pid);

    /**
     * Subject constructor, it creates UnixSession object from PolkitUnixSession object
     *
     * \warning Use this only if you are completely aware of what are you doing!
     *
     * \param pkUnixSession PolkitUnixSession object
     */
    explicit UnixSessionSubject(PolkitSystemBusName *pkUnixSession);

    /**
    * Returns session id.
    *
    * \return A session id.
    */
    TQString sessionId() const;

    /**
    * Sets session id.
    *
    * \param sessionId A session id.
    */
    void setSessionId(const TQString &sessionId);
};

}

#endif
