#!/bin/sh
# SPDX-License-Identifier: BSD-2-Clause
# SPDX-FileCopyrightText: © 2025 Antoine Le Gonidec <vv221@dotslashplay.it>
set -o errexit

###
# Age of Empires 2
# send your bug reports to vv221@dotslashplay.it
###

script_version=20251104.2

PLAYIT_COMPATIBILITY_LEVEL='2.33'

GAME_ID='age-of-empires-2'
GAME_NAME='Age of Empires II: The Age of Kings'

ARCHIVE_BASE_FR_0_NAME='AOE2.iso'
ARCHIVE_BASE_FR_0_MD5='d5ff1494615233c7eb3d6ddc61930a40'
ARCHIVE_BASE_FR_0_SIZE='483970'
ARCHIVE_BASE_FR_0_VERSION='1.0-1'

ARCHIVE_BASE_EN_0_NAME='AOE2.iso'
ARCHIVE_BASE_EN_0_MD5='8153250ca4c15b1979c3899f986f2627'
ARCHIVE_BASE_EN_0_SIZE='471713'
ARCHIVE_BASE_EN_0_VERSION='1.0-1'

ARCHIVE_OPTIONAL_FR_PATCH_NAME='Age2upA-FRN1.exe'
ARCHIVE_OPTIONAL_FR_PATCH_MD5='f632bf26464cad1212588fc817599bae'
ARCHIVE_OPTIONAL_FR_PATCH_TYPE='cabinet'
ARCHIVE_OPTIONAL_FR_PATCH_SIZE='3849'
ARCHIVE_OPTIONAL_FR_PATCH_VERSION='2.0c-1'

ARCHIVE_OPTIONAL_EN_PATCH_NAME='Age2upA.exe'
ARCHIVE_OPTIONAL_EN_PATCH_MD5='20b9fd4ebd51d0375882f4f510ca7c36'
ARCHIVE_OPTIONAL_EN_PATCH_TYPE='cabinet'
ARCHIVE_OPTIONAL_EN_PATCH_SIZE='3843'
ARCHIVE_OPTIONAL_EN_PATCH_VERSION='2.0c-1'

## Files found directly on the CD-ROM

CONTENT_PATH_DEFAULT='GAME'
## The game crashes on launch if the following library is missing.
CONTENT_LIBS_BIN_FILES='
language.dll'
CONTENT_GAME_BIN_FILES='
empires2.exe
empires2.icd'
CONTENT_GAME_L10N_FILES='
campaign/media/backgrd*
campaign/media/c*
sound/campaign'
CONTENT_GAME_L10N_COMMON_FILES='
campaign/media/intro*'
CONTENT_GAME_DATA_FILES='
avi
data
sound/stream'

## Files found in the cabinet archive

## The game crashes on launch if the following library is missing.
CONTENT_LIBS0_BIN_PATH='.'
CONTENT_LIBS0_BIN_FILES='
ebueula.dll'
CONTENT_GAME0_BIN_PATH='.'
CONTENT_GAME0_BIN_FILES='
data/closedpw.exe'
CONTENT_GAME0_L10N_PATH='.'
CONTENT_GAME0_L10N_FILES="
data/list.cr
history
sound/scenario
taunt/01 yes.mp3
taunt/02 no.mp3
taunt/07 ahh.mp3
taunt/08 all hail.mp3
taunt/09 oooh.mp3
taunt/10 back to age 1.mp3
taunt/11 herb laugh.mp3
taunt/12 being rushed.mp3
taunt/13 blame your isp.mp3
taunt/14 start the game.mp3
taunt/15 don't point that thing.mp3
taunt/16 enemy sighted.mp3
taunt/17 it is good.mp3
taunt/18 i need a monk.mp3
taunt/19 long time no siege.mp3
taunt/20 my granny.mp3
taunt/21 nice town i'll take it.mp3
taunt/22 quit touchin.mp3
taunt/23 raiding party.mp3
taunt/24 dadgum.mp3
taunt/25 smite me.mp3
taunt/26 the wonder.mp3
taunt/27 you play 2 hours.mp3
taunt/28 you should see the other guy.mp3
taunt/29 roggan.mp3
taunt/30 wololo.mp3"
CONTENT_GAME0_L10N_COMMON_PATH='.'
CONTENT_GAME0_L10N_COMMON_FILES='
taunt/03 food, please.mp3
taunt/04 wood, please.mp3
taunt/05 gold, please.mp3
taunt/06 stone, please.mp3'
CONTENT_GAME0_DATA_PATH='.'
CONTENT_GAME0_DATA_FILES='
ai
campaign
data/load
data/blkedge.dat
data/filtermaps.dat
data/lightmaps.dat
data/loqmaps.dat
data/patternmasks.dat
data/stemplet.dat
data/tileedge.dat
data/gamedata.drs
data/sounds.drs
learn
savegame
scenario
sound/terrain
scenariobkg.bmp'
CONTENT_GAME_DATA_COMMON_PATH='.'
CONTENT_GAME_DATA_COMMON_FILES='
data/shadow.col
data/blendomatic.dat
data/view_icm.dat
data/graphics.drs
data/interfac.drs
data/terrain.drs'

## Files found in the 2.0c patch

CONTENT_GAME1_BIN_PATH='.'
CONTENT_GAME1_BIN_FILES='
empires2.exe
empires2.icd'
CONTENT_GAME1_DATA_PATH='.'
CONTENT_GAME1_DATA_FILES='
data/gamedata.drs'

USER_PERSISTENT_DIRECTORIES='
campaign/*.cpn
learn
savegame
scenario'
USER_PERSISTENT_FILES='
data/*.dat
*.nfp'

## Without a WINE virtual desktop, the game menu is badly rendered.
## (WINE 10.0)
WINE_VIRTUAL_DESKTOP='1280x1024'

APP_MAIN_EXE='empires2.exe'
APP_MAIN_ICON='../48256.ICO'

PACKAGES_LIST='
PKG_BIN
PKG_L10N
PKG_L10N_COMMON
PKG_DATA
PKG_DATA_COMMON'

PKG_BIN_ID="$GAME_ID"
PKG_BIN_ID_FR="${PKG_BIN_ID}-fr"
PKG_BIN_ID_EN="${PKG_BIN_ID}-en"
PKG_BIN_PROVIDES="
$PKG_BIN_ID"
PKG_BIN_ARCH='32'
PKG_BIN_DEPENDENCIES_SIBLINGS='
PKG_L10N
PKG_L10N_COMMON
PKG_DATA'
PKG_BIN_DEPENDENCIES_GSTREAMER_PLUGINS='
video/x-msvideo'

PKG_L10N_ID="${GAME_ID}-l10n"
PKG_L10N_ID_FR="${PKG_L10N_ID}-fr"
PKG_L10N_ID_EN="${PKG_L10N_ID}-en"
PKG_L10N_PROVIDES="
$PKG_L10N_ID"
PKG_L10N_DESCRIPTION_FR='French localisation'
PKG_L10N_DESCRIPTION_EN='English localisation'
PKG_L10N_DEPENDENCIES_SIBLINGS='
PKG_L10N_COMMON'

PKG_L10N_COMMON_ID="${PKG_L10N_ID}-common"
PKG_L10N_COMMON_ID_FR="${PKG_L10N_COMMON_ID}-fr"
PKG_L10N_COMMON_ID_EN="${PKG_L10N_COMMON_ID}-en"
PKG_L10N_COMMON_PROVIDES="
$PKG_L10N_COMMON_ID"
PKG_L10N_COMMON_DESCRIPTION_FR='French localisation - files shared between the base game and the expansion'
PKG_L10N_COMMON_DESCRIPTION_EN='English localisation - files shared between the base game and the expansion'

PKG_DATA_ID="${GAME_ID}-data"
PKG_DATA_DESCRIPTION='data'
PKG_DATA_DEPENDENCIES_SIBLINGS='
PKG_DATA_COMMON'

PKG_DATA_COMMON_ID="${PKG_DATA_ID}-common"
PKG_DATA_COMMON_DESCRIPTION='data - files shared between the base game and the expansion'

## - cabextract is used to extract files from a cabinet archive on the CD-ROM.
## - convmv is used to convert file paths to lower case.
REQUIREMENTS_LIST='
cabextract
convmv'

# Load common functions.

PLAYIT_LIB_PATHS="
$PWD
${XDG_DATA_HOME:="${HOME}/.local/share"}/play.it
/usr/local/share/games/play.it
/usr/local/share/play.it
/usr/share/games/play.it
/usr/share/play.it"

if [ -z "$PLAYIT_LIB2" ]; then
	for playit_lib_path in $PLAYIT_LIB_PATHS; do
		if [ -e "${playit_lib_path}/libplayit2.sh" ]; then
			PLAYIT_LIB2="${playit_lib_path}/libplayit2.sh"
			break
		fi
	done
fi
if [ -z "$PLAYIT_LIB2" ]; then
	printf '\n\033[1;31mError:\033[0m\n'
	printf 'libplayit2.sh not found.\n'
	exit 1
fi
# shellcheck source=libplayit2.sh
. "$PLAYIT_LIB2"

# Run default initialisation actions.

initialization_default "$@"

# Check for the presence of the 2.0c patch.

case "$(current_archive)" in
	('ARCHIVE_BASE_FR_'*)
		archive_initialize_optional 'ARCHIVE_PATCH' 'ARCHIVE_OPTIONAL_FR_PATCH'
	;;
	('ARCHIVE_BASE_EN_'*)
		archive_initialize_optional 'ARCHIVE_PATCH' 'ARCHIVE_OPTIONAL_EN_PATCH'
	;;
esac
if archive_is_available 'ARCHIVE_PATCH'; then
	export "$(current_archive)_VERSION=$(archive_version 'ARCHIVE_PATCH')"
fi

# Extract game data.

archive_extraction_default
## Convert all file paths to lowercase.
tolower "${PLAYIT_WORKDIR}/gamedata/$(content_path_default)"
## Extract files from the cabinet archive.
ARCHIVE_CABINET_PATH="${PLAYIT_WORKDIR}/gamedata/$(content_path_default)/empires2.cab"
archive_extraction 'ARCHIVE_CABINET'
rm "$ARCHIVE_CABINET_PATH"
## Extract files from the 2.0c patch.
if archive_is_available 'ARCHIVE_PATCH'; then
	archive_extraction 'ARCHIVE_PATCH'
	case "$(current_archive)" in
		('ARCHIVE_BASE_FR_'*)
			mv \
				"${PLAYIT_WORKDIR}/gamedata/$(content_path 'GAME1_BIN')/empires2frn.exe" \
				"${PLAYIT_WORKDIR}/gamedata/$(content_path 'GAME1_BIN')/empires2.exe"
			mv \
				"${PLAYIT_WORKDIR}/gamedata/$(content_path 'GAME1_BIN')/empires2frn.icd" \
				"${PLAYIT_WORKDIR}/gamedata/$(content_path 'GAME1_BIN')/empires2.icd"
		;;
		('ARCHIVE_BASE_EN_'*)
			mv \
				"${PLAYIT_WORKDIR}/gamedata/$(content_path 'GAME1_BIN')/empires2usa.exe" \
				"${PLAYIT_WORKDIR}/gamedata/$(content_path 'GAME1_BIN')/empires2.exe"
			mv \
				"${PLAYIT_WORKDIR}/gamedata/$(content_path 'GAME1_BIN')/empires2usa.icd" \
				"${PLAYIT_WORKDIR}/gamedata/$(content_path 'GAME1_BIN')/empires2.icd"
		;;
	esac
fi

# Include game data.

content_inclusion_icons 'PKG_DATA'
content_inclusion_default

# Write launchers.

## Install shipped libraries.
wineprefix_init_custom() {
	cat <<- EOF
	# Install shipped libraries.
	PATH_LIBRARIES_SYSTEM='$(path_libraries)'
	EOF
	cat <<- 'EOF'
	find "$PATH_LIBRARIES_SYSTEM" -iname '*.dll' | while read -r dll_source; do
		dll_destination="${PATH_PREFIX}${dll_source#"${PATH_LIBRARIES_SYSTEM}"}"
	    mkdir --parents "$(dirname "$dll_destination")"
	    ln --symbolic "$dll_source" "$dll_destination"
	done
	EOF
}

launchers_generation

# Build packages.

packages_generation
print_instructions

# Clean up.

working_directory_cleanup

exit 0
