{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Instances of the @GDBusMethodInvocation@ class are used when
-- handling D-Bus method calls. It provides a way to asynchronously
-- return results and errors.
-- 
-- The normal way to obtain a @GDBusMethodInvocation@ object is to receive
-- it as an argument to the @handle_method_call()@ function in a
-- [type/@gio@/.DBusInterfaceVTable] that was passed to
-- 'GI.Gio.Objects.DBusConnection.dBusConnectionRegisterObject'.
-- 
-- /Since: 2.26/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.DBusMethodInvocation
    ( 

-- * Exported types
    DBusMethodInvocation(..)                ,
    IsDBusMethodInvocation                  ,
    toDBusMethodInvocation                  ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [returnDbusError]("GI.Gio.Objects.DBusMethodInvocation#g:method:returnDbusError"), [returnErrorLiteral]("GI.Gio.Objects.DBusMethodInvocation#g:method:returnErrorLiteral"), [returnGerror]("GI.Gio.Objects.DBusMethodInvocation#g:method:returnGerror"), [returnValue]("GI.Gio.Objects.DBusMethodInvocation#g:method:returnValue"), [returnValueWithUnixFdList]("GI.Gio.Objects.DBusMethodInvocation#g:method:returnValueWithUnixFdList"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getConnection]("GI.Gio.Objects.DBusMethodInvocation#g:method:getConnection"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getInterfaceName]("GI.Gio.Objects.DBusMethodInvocation#g:method:getInterfaceName"), [getMessage]("GI.Gio.Objects.DBusMethodInvocation#g:method:getMessage"), [getMethodInfo]("GI.Gio.Objects.DBusMethodInvocation#g:method:getMethodInfo"), [getMethodName]("GI.Gio.Objects.DBusMethodInvocation#g:method:getMethodName"), [getObjectPath]("GI.Gio.Objects.DBusMethodInvocation#g:method:getObjectPath"), [getParameters]("GI.Gio.Objects.DBusMethodInvocation#g:method:getParameters"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getPropertyInfo]("GI.Gio.Objects.DBusMethodInvocation#g:method:getPropertyInfo"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getSender]("GI.Gio.Objects.DBusMethodInvocation#g:method:getSender").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveDBusMethodInvocationMethod       ,
#endif

-- ** getConnection #method:getConnection#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationGetConnectionMethodInfo,
#endif
    dBusMethodInvocationGetConnection       ,


-- ** getInterfaceName #method:getInterfaceName#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationGetInterfaceNameMethodInfo,
#endif
    dBusMethodInvocationGetInterfaceName    ,


-- ** getMessage #method:getMessage#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationGetMessageMethodInfo,
#endif
    dBusMethodInvocationGetMessage          ,


-- ** getMethodInfo #method:getMethodInfo#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationGetMethodInfoMethodInfo,
#endif
    dBusMethodInvocationGetMethodInfo       ,


-- ** getMethodName #method:getMethodName#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationGetMethodNameMethodInfo,
#endif
    dBusMethodInvocationGetMethodName       ,


-- ** getObjectPath #method:getObjectPath#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationGetObjectPathMethodInfo,
#endif
    dBusMethodInvocationGetObjectPath       ,


-- ** getParameters #method:getParameters#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationGetParametersMethodInfo,
#endif
    dBusMethodInvocationGetParameters       ,


-- ** getPropertyInfo #method:getPropertyInfo#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationGetPropertyInfoMethodInfo,
#endif
    dBusMethodInvocationGetPropertyInfo     ,


-- ** getSender #method:getSender#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationGetSenderMethodInfo ,
#endif
    dBusMethodInvocationGetSender           ,


-- ** returnDbusError #method:returnDbusError#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationReturnDbusErrorMethodInfo,
#endif
    dBusMethodInvocationReturnDbusError     ,


-- ** returnErrorLiteral #method:returnErrorLiteral#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationReturnErrorLiteralMethodInfo,
#endif
    dBusMethodInvocationReturnErrorLiteral  ,


-- ** returnGerror #method:returnGerror#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationReturnGerrorMethodInfo,
#endif
    dBusMethodInvocationReturnGerror        ,


-- ** returnValue #method:returnValue#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationReturnValueMethodInfo,
#endif
    dBusMethodInvocationReturnValue         ,


-- ** returnValueWithUnixFdList #method:returnValueWithUnixFdList#

#if defined(ENABLE_OVERLOADING)
    DBusMethodInvocationReturnValueWithUnixFdListMethodInfo,
#endif
    dBusMethodInvocationReturnValueWithUnixFdList,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GLib.Structs.String as GLib.String
import qualified GI.GLib.Structs.VariantType as GLib.VariantType
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.Parameter as GObject.Parameter
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncInitable as Gio.AsyncInitable
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.Credentials as Gio.Credentials
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusAuthObserver as Gio.DBusAuthObserver
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusConnection as Gio.DBusConnection
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusMessage as Gio.DBusMessage
import {-# SOURCE #-} qualified GI.Gio.Objects.IOStream as Gio.IOStream
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuAttributeIter as Gio.MenuAttributeIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuLinkIter as Gio.MenuLinkIter
import {-# SOURCE #-} qualified GI.Gio.Objects.MenuModel as Gio.MenuModel
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.UnixFDList as Gio.UnixFDList
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusAnnotationInfo as Gio.DBusAnnotationInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusArgInfo as Gio.DBusArgInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusInterfaceInfo as Gio.DBusInterfaceInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusMethodInfo as Gio.DBusMethodInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusPropertyInfo as Gio.DBusPropertyInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusSignalInfo as Gio.DBusSignalInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusSubtreeVTable as Gio.DBusSubtreeVTable
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusConnection as Gio.DBusConnection
import {-# SOURCE #-} qualified GI.Gio.Objects.DBusMessage as Gio.DBusMessage
import {-# SOURCE #-} qualified GI.Gio.Objects.UnixFDList as Gio.UnixFDList
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusMethodInfo as Gio.DBusMethodInfo
import {-# SOURCE #-} qualified GI.Gio.Structs.DBusPropertyInfo as Gio.DBusPropertyInfo

#endif

-- | Memory-managed wrapper type.
newtype DBusMethodInvocation = DBusMethodInvocation (SP.ManagedPtr DBusMethodInvocation)
    deriving (Eq)

instance SP.ManagedPtrNewtype DBusMethodInvocation where
    toManagedPtr (DBusMethodInvocation p) = p

foreign import ccall "g_dbus_method_invocation_get_type"
    c_g_dbus_method_invocation_get_type :: IO B.Types.GType

instance B.Types.TypedObject DBusMethodInvocation where
    glibType = c_g_dbus_method_invocation_get_type

instance B.Types.GObject DBusMethodInvocation

-- | Type class for types which can be safely cast to t'DBusMethodInvocation', for instance with `toDBusMethodInvocation`.
class (SP.GObject o, O.IsDescendantOf DBusMethodInvocation o) => IsDBusMethodInvocation o
instance (SP.GObject o, O.IsDescendantOf DBusMethodInvocation o) => IsDBusMethodInvocation o

instance O.HasParentTypes DBusMethodInvocation
type instance O.ParentTypes DBusMethodInvocation = '[GObject.Object.Object]

-- | Cast to t'DBusMethodInvocation', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toDBusMethodInvocation :: (MIO.MonadIO m, IsDBusMethodInvocation o) => o -> m DBusMethodInvocation
toDBusMethodInvocation = MIO.liftIO . B.ManagedPtr.unsafeCastTo DBusMethodInvocation

-- | Convert t'DBusMethodInvocation' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe DBusMethodInvocation) where
    gvalueGType_ = c_g_dbus_method_invocation_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr DBusMethodInvocation)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr DBusMethodInvocation)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject DBusMethodInvocation ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveDBusMethodInvocationMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveDBusMethodInvocationMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveDBusMethodInvocationMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveDBusMethodInvocationMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveDBusMethodInvocationMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveDBusMethodInvocationMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveDBusMethodInvocationMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveDBusMethodInvocationMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveDBusMethodInvocationMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveDBusMethodInvocationMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveDBusMethodInvocationMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveDBusMethodInvocationMethod "returnDbusError" o = DBusMethodInvocationReturnDbusErrorMethodInfo
    ResolveDBusMethodInvocationMethod "returnErrorLiteral" o = DBusMethodInvocationReturnErrorLiteralMethodInfo
    ResolveDBusMethodInvocationMethod "returnGerror" o = DBusMethodInvocationReturnGerrorMethodInfo
    ResolveDBusMethodInvocationMethod "returnValue" o = DBusMethodInvocationReturnValueMethodInfo
    ResolveDBusMethodInvocationMethod "returnValueWithUnixFdList" o = DBusMethodInvocationReturnValueWithUnixFdListMethodInfo
    ResolveDBusMethodInvocationMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveDBusMethodInvocationMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveDBusMethodInvocationMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveDBusMethodInvocationMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveDBusMethodInvocationMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveDBusMethodInvocationMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveDBusMethodInvocationMethod "getConnection" o = DBusMethodInvocationGetConnectionMethodInfo
    ResolveDBusMethodInvocationMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveDBusMethodInvocationMethod "getInterfaceName" o = DBusMethodInvocationGetInterfaceNameMethodInfo
    ResolveDBusMethodInvocationMethod "getMessage" o = DBusMethodInvocationGetMessageMethodInfo
    ResolveDBusMethodInvocationMethod "getMethodInfo" o = DBusMethodInvocationGetMethodInfoMethodInfo
    ResolveDBusMethodInvocationMethod "getMethodName" o = DBusMethodInvocationGetMethodNameMethodInfo
    ResolveDBusMethodInvocationMethod "getObjectPath" o = DBusMethodInvocationGetObjectPathMethodInfo
    ResolveDBusMethodInvocationMethod "getParameters" o = DBusMethodInvocationGetParametersMethodInfo
    ResolveDBusMethodInvocationMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveDBusMethodInvocationMethod "getPropertyInfo" o = DBusMethodInvocationGetPropertyInfoMethodInfo
    ResolveDBusMethodInvocationMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveDBusMethodInvocationMethod "getSender" o = DBusMethodInvocationGetSenderMethodInfo
    ResolveDBusMethodInvocationMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveDBusMethodInvocationMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveDBusMethodInvocationMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveDBusMethodInvocationMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveDBusMethodInvocationMethod t DBusMethodInvocation, O.OverloadedMethod info DBusMethodInvocation p) => OL.IsLabel t (DBusMethodInvocation -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveDBusMethodInvocationMethod t DBusMethodInvocation, O.OverloadedMethod info DBusMethodInvocation p, R.HasField t DBusMethodInvocation p) => R.HasField t DBusMethodInvocation p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveDBusMethodInvocationMethod t DBusMethodInvocation, O.OverloadedMethodInfo info DBusMethodInvocation) => OL.IsLabel t (O.MethodProxy info DBusMethodInvocation) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList DBusMethodInvocation
type instance O.AttributeList DBusMethodInvocation = DBusMethodInvocationAttributeList
type DBusMethodInvocationAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList DBusMethodInvocation = DBusMethodInvocationSignalList
type DBusMethodInvocationSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method DBusMethodInvocation::get_connection
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "DBusConnection" })
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_get_connection" g_dbus_method_invocation_get_connection :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    IO (Ptr Gio.DBusConnection.DBusConnection)

-- | Gets the t'GI.Gio.Objects.DBusConnection.DBusConnection' the method was invoked on.
-- 
-- /Since: 2.26/
dBusMethodInvocationGetConnection ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> m Gio.DBusConnection.DBusConnection
    -- ^ __Returns:__ A t'GI.Gio.Objects.DBusConnection.DBusConnection'. Do not free, it is owned by /@invocation@/.
dBusMethodInvocationGetConnection invocation = liftIO $ do
    invocation' <- unsafeManagedPtrCastPtr invocation
    result <- g_dbus_method_invocation_get_connection invocation'
    checkUnexpectedReturnNULL "dBusMethodInvocationGetConnection" result
    result' <- (newObject Gio.DBusConnection.DBusConnection) result
    touchManagedPtr invocation
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationGetConnectionMethodInfo
instance (signature ~ (m Gio.DBusConnection.DBusConnection), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationGetConnectionMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationGetConnection

instance O.OverloadedMethodInfo DBusMethodInvocationGetConnectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetConnection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationGetConnection"
        })


#endif

-- method DBusMethodInvocation::get_interface_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_get_interface_name" g_dbus_method_invocation_get_interface_name :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    IO CString

-- | Gets the name of the D-Bus interface the method was invoked on.
-- 
-- This can be @NULL@ if it was not specified by the sender. See
-- [callback/@gio@/.DBusInterfaceMethodCallFunc] or the
-- <https://dbus.freedesktop.org/doc/dbus-specification.html#message-protocol-types-method D-Bus Specification>
-- for details on when this can happen and how it should be handled.
-- 
-- If this method call is a property Get, Set or GetAll call that has
-- been redirected to the method call handler then
-- \"org.freedesktop.DBus.Properties\" will be returned.  See
-- t'GI.Gio.Structs.DBusInterfaceVTable.DBusInterfaceVTable' for more information.
-- 
-- /Since: 2.26/
dBusMethodInvocationGetInterfaceName ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ A string. Do not free, it is owned by /@invocation@/.
dBusMethodInvocationGetInterfaceName invocation = liftIO $ do
    invocation' <- unsafeManagedPtrCastPtr invocation
    result <- g_dbus_method_invocation_get_interface_name invocation'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr invocation
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationGetInterfaceNameMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationGetInterfaceNameMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationGetInterfaceName

instance O.OverloadedMethodInfo DBusMethodInvocationGetInterfaceNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetInterfaceName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationGetInterfaceName"
        })


#endif

-- method DBusMethodInvocation::get_message
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "DBusMessage" })
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_get_message" g_dbus_method_invocation_get_message :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    IO (Ptr Gio.DBusMessage.DBusMessage)

-- | Gets the t'GI.Gio.Objects.DBusMessage.DBusMessage' for the method invocation. This is useful if
-- you need to use low-level protocol features, such as UNIX file
-- descriptor passing, that cannot be properly expressed in the
-- t'GVariant' API.
-- 
-- See this [server][class/@gio@/.DBusConnection@/an/@-example-d-bus-server]
-- and [client][class/@gio@/.DBusConnection@/an/@-example-for-file-descriptor-passing]
-- for an example of how to use this low-level API to send and receive
-- UNIX file descriptors.
-- 
-- /Since: 2.26/
dBusMethodInvocationGetMessage ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> m Gio.DBusMessage.DBusMessage
    -- ^ __Returns:__ t'GI.Gio.Objects.DBusMessage.DBusMessage'. Do not free, it is owned by /@invocation@/.
dBusMethodInvocationGetMessage invocation = liftIO $ do
    invocation' <- unsafeManagedPtrCastPtr invocation
    result <- g_dbus_method_invocation_get_message invocation'
    checkUnexpectedReturnNULL "dBusMethodInvocationGetMessage" result
    result' <- (newObject Gio.DBusMessage.DBusMessage) result
    touchManagedPtr invocation
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationGetMessageMethodInfo
instance (signature ~ (m Gio.DBusMessage.DBusMessage), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationGetMessageMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationGetMessage

instance O.OverloadedMethodInfo DBusMethodInvocationGetMessageMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetMessage",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationGetMessage"
        })


#endif

-- method DBusMethodInvocation::get_method_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "DBusMethodInfo" })
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_get_method_info" g_dbus_method_invocation_get_method_info :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    IO (Ptr Gio.DBusMethodInfo.DBusMethodInfo)

-- | Gets information about the method call, if any.
-- 
-- If this method invocation is a property Get, Set or GetAll call that
-- has been redirected to the method call handler then 'P.Nothing' will be
-- returned.  See 'GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetPropertyInfo' and
-- t'GI.Gio.Structs.DBusInterfaceVTable.DBusInterfaceVTable' for more information.
-- 
-- /Since: 2.26/
dBusMethodInvocationGetMethodInfo ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> m (Maybe Gio.DBusMethodInfo.DBusMethodInfo)
    -- ^ __Returns:__ A t'GI.Gio.Structs.DBusMethodInfo.DBusMethodInfo' or 'P.Nothing'. Do not free, it is owned by /@invocation@/.
dBusMethodInvocationGetMethodInfo invocation = liftIO $ do
    invocation' <- unsafeManagedPtrCastPtr invocation
    result <- g_dbus_method_invocation_get_method_info invocation'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Gio.DBusMethodInfo.DBusMethodInfo) result'
        return result''
    touchManagedPtr invocation
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationGetMethodInfoMethodInfo
instance (signature ~ (m (Maybe Gio.DBusMethodInfo.DBusMethodInfo)), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationGetMethodInfoMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationGetMethodInfo

instance O.OverloadedMethodInfo DBusMethodInvocationGetMethodInfoMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetMethodInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationGetMethodInfo"
        })


#endif

-- method DBusMethodInvocation::get_method_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_get_method_name" g_dbus_method_invocation_get_method_name :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    IO CString

-- | Gets the name of the method that was invoked.
-- 
-- /Since: 2.26/
dBusMethodInvocationGetMethodName ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> m T.Text
    -- ^ __Returns:__ A string. Do not free, it is owned by /@invocation@/.
dBusMethodInvocationGetMethodName invocation = liftIO $ do
    invocation' <- unsafeManagedPtrCastPtr invocation
    result <- g_dbus_method_invocation_get_method_name invocation'
    checkUnexpectedReturnNULL "dBusMethodInvocationGetMethodName" result
    result' <- cstringToText result
    touchManagedPtr invocation
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationGetMethodNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationGetMethodNameMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationGetMethodName

instance O.OverloadedMethodInfo DBusMethodInvocationGetMethodNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetMethodName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationGetMethodName"
        })


#endif

-- method DBusMethodInvocation::get_object_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_get_object_path" g_dbus_method_invocation_get_object_path :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    IO CString

-- | Gets the object path the method was invoked on.
-- 
-- /Since: 2.26/
dBusMethodInvocationGetObjectPath ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> m T.Text
    -- ^ __Returns:__ A string. Do not free, it is owned by /@invocation@/.
dBusMethodInvocationGetObjectPath invocation = liftIO $ do
    invocation' <- unsafeManagedPtrCastPtr invocation
    result <- g_dbus_method_invocation_get_object_path invocation'
    checkUnexpectedReturnNULL "dBusMethodInvocationGetObjectPath" result
    result' <- cstringToText result
    touchManagedPtr invocation
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationGetObjectPathMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationGetObjectPathMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationGetObjectPath

instance O.OverloadedMethodInfo DBusMethodInvocationGetObjectPathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetObjectPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationGetObjectPath"
        })


#endif

-- method DBusMethodInvocation::get_parameters
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TVariant
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_get_parameters" g_dbus_method_invocation_get_parameters :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    IO (Ptr GVariant)

-- | Gets the parameters of the method invocation. If there are no input
-- parameters then this will return a GVariant with 0 children rather than NULL.
-- 
-- /Since: 2.26/
dBusMethodInvocationGetParameters ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> m GVariant
    -- ^ __Returns:__ A t'GVariant' tuple. Do not unref this because it is owned by /@invocation@/.
dBusMethodInvocationGetParameters invocation = liftIO $ do
    invocation' <- unsafeManagedPtrCastPtr invocation
    result <- g_dbus_method_invocation_get_parameters invocation'
    checkUnexpectedReturnNULL "dBusMethodInvocationGetParameters" result
    result' <- B.GVariant.newGVariantFromPtr result
    touchManagedPtr invocation
    return result'

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationGetParametersMethodInfo
instance (signature ~ (m GVariant), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationGetParametersMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationGetParameters

instance O.OverloadedMethodInfo DBusMethodInvocationGetParametersMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetParameters",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationGetParameters"
        })


#endif

-- method DBusMethodInvocation::get_property_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "DBusPropertyInfo" })
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_get_property_info" g_dbus_method_invocation_get_property_info :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    IO (Ptr Gio.DBusPropertyInfo.DBusPropertyInfo)

-- | Gets information about the property that this method call is for, if
-- any.
-- 
-- This will only be set in the case of an invocation in response to a
-- property Get or Set call that has been directed to the method call
-- handler for an object on account of its @/property_get()/@ or
-- @/property_set()/@ vtable pointers being unset.
-- 
-- See t'GI.Gio.Structs.DBusInterfaceVTable.DBusInterfaceVTable' for more information.
-- 
-- If the call was GetAll, 'P.Nothing' will be returned.
-- 
-- /Since: 2.38/
dBusMethodInvocationGetPropertyInfo ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'
    -> m (Maybe Gio.DBusPropertyInfo.DBusPropertyInfo)
    -- ^ __Returns:__ a t'GI.Gio.Structs.DBusPropertyInfo.DBusPropertyInfo' or 'P.Nothing'
dBusMethodInvocationGetPropertyInfo invocation = liftIO $ do
    invocation' <- unsafeManagedPtrCastPtr invocation
    result <- g_dbus_method_invocation_get_property_info invocation'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Gio.DBusPropertyInfo.DBusPropertyInfo) result'
        return result''
    touchManagedPtr invocation
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationGetPropertyInfoMethodInfo
instance (signature ~ (m (Maybe Gio.DBusPropertyInfo.DBusPropertyInfo)), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationGetPropertyInfoMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationGetPropertyInfo

instance O.OverloadedMethodInfo DBusMethodInvocationGetPropertyInfoMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetPropertyInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationGetPropertyInfo"
        })


#endif

-- method DBusMethodInvocation::get_sender
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_get_sender" g_dbus_method_invocation_get_sender :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    IO CString

-- | Gets the bus name that invoked the method.
-- 
-- This can return 'P.Nothing' if not specified by the caller, e.g. on peer-to-peer
-- connections.
-- 
-- /Since: 2.26/
dBusMethodInvocationGetSender ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> m (Maybe T.Text)
    -- ^ __Returns:__ A string. Do not free, it is owned by /@invocation@/.
dBusMethodInvocationGetSender invocation = liftIO $ do
    invocation' <- unsafeManagedPtrCastPtr invocation
    result <- g_dbus_method_invocation_get_sender invocation'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr invocation
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationGetSenderMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationGetSenderMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationGetSender

instance O.OverloadedMethodInfo DBusMethodInvocationGetSenderMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationGetSender",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationGetSender"
        })


#endif

-- method DBusMethodInvocation::return_dbus_error
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "error_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A valid D-Bus error name."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "error_message"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A valid D-Bus error message."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_return_dbus_error" g_dbus_method_invocation_return_dbus_error :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    CString ->                              -- error_name : TBasicType TUTF8
    CString ->                              -- error_message : TBasicType TUTF8
    IO ()

-- | Finishes handling a D-Bus method call by returning an error.
-- 
-- This method will take ownership of /@invocation@/. See
-- t'GI.Gio.Structs.DBusInterfaceVTable.DBusInterfaceVTable' for more information about the ownership of
-- /@invocation@/.
-- 
-- /Since: 2.26/
dBusMethodInvocationReturnDbusError ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> T.Text
    -- ^ /@errorName@/: A valid D-Bus error name.
    -> T.Text
    -- ^ /@errorMessage@/: A valid D-Bus error message.
    -> m ()
dBusMethodInvocationReturnDbusError invocation errorName errorMessage = liftIO $ do
    invocation' <- B.ManagedPtr.disownObject invocation
    errorName' <- textToCString errorName
    errorMessage' <- textToCString errorMessage
    g_dbus_method_invocation_return_dbus_error invocation' errorName' errorMessage'
    touchManagedPtr invocation
    freeMem errorName'
    freeMem errorMessage'
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationReturnDbusErrorMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ()), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationReturnDbusErrorMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationReturnDbusError

instance O.OverloadedMethodInfo DBusMethodInvocationReturnDbusErrorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationReturnDbusError",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationReturnDbusError"
        })


#endif

-- method DBusMethodInvocation::return_error_literal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "domain"
--           , argType = TBasicType TUInt32
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GQuark for the #GError error domain."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "code"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The error code." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "message"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The error message." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_return_error_literal" g_dbus_method_invocation_return_error_literal :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    Word32 ->                               -- domain : TBasicType TUInt32
    Int32 ->                                -- code : TBasicType TInt
    CString ->                              -- message : TBasicType TUTF8
    IO ()

-- | Like @/g_dbus_method_invocation_return_error()/@ but without @/printf()/@-style formatting.
-- 
-- This method will take ownership of /@invocation@/. See
-- t'GI.Gio.Structs.DBusInterfaceVTable.DBusInterfaceVTable' for more information about the ownership of
-- /@invocation@/.
-- 
-- /Since: 2.26/
dBusMethodInvocationReturnErrorLiteral ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> Word32
    -- ^ /@domain@/: A @/GQuark/@ for the t'GError' error domain.
    -> Int32
    -- ^ /@code@/: The error code.
    -> T.Text
    -- ^ /@message@/: The error message.
    -> m ()
dBusMethodInvocationReturnErrorLiteral invocation domain code message = liftIO $ do
    invocation' <- B.ManagedPtr.disownObject invocation
    message' <- textToCString message
    g_dbus_method_invocation_return_error_literal invocation' domain code message'
    touchManagedPtr invocation
    freeMem message'
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationReturnErrorLiteralMethodInfo
instance (signature ~ (Word32 -> Int32 -> T.Text -> m ()), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationReturnErrorLiteralMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationReturnErrorLiteral

instance O.OverloadedMethodInfo DBusMethodInvocationReturnErrorLiteralMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationReturnErrorLiteral",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationReturnErrorLiteral"
        })


#endif

-- method DBusMethodInvocation::return_gerror
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "error"
--           , argType = TError
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GError." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_return_gerror" g_dbus_method_invocation_return_gerror :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    Ptr GError ->                           -- error : TError
    IO ()

-- | Like @/g_dbus_method_invocation_return_error()/@ but takes a t'GError'
-- instead of the error domain, error code and message.
-- 
-- This method will take ownership of /@invocation@/. See
-- t'GI.Gio.Structs.DBusInterfaceVTable.DBusInterfaceVTable' for more information about the ownership of
-- /@invocation@/.
-- 
-- /Since: 2.26/
dBusMethodInvocationReturnGerror ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> GError
    -- ^ /@error@/: A t'GError'.
    -> m ()
dBusMethodInvocationReturnGerror invocation error_ = liftIO $ do
    invocation' <- B.ManagedPtr.disownObject invocation
    error_' <- unsafeManagedPtrGetPtr error_
    g_dbus_method_invocation_return_gerror invocation' error_'
    touchManagedPtr invocation
    touchManagedPtr error_
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationReturnGerrorMethodInfo
instance (signature ~ (GError -> m ()), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationReturnGerrorMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationReturnGerror

instance O.OverloadedMethodInfo DBusMethodInvocationReturnGerrorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationReturnGerror",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationReturnGerror"
        })


#endif

-- method DBusMethodInvocation::return_value
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "parameters"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GVariant tuple with out parameters for the method or %NULL if not passing any parameters."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_return_value" g_dbus_method_invocation_return_value :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    Ptr GVariant ->                         -- parameters : TVariant
    IO ()

-- | Finishes handling a D-Bus method call by returning /@parameters@/.
-- If the /@parameters@/ GVariant is floating, it is consumed.
-- 
-- It is an error if /@parameters@/ is not of the right format: it must be a tuple
-- containing the out-parameters of the D-Bus method. Even if the method has a
-- single out-parameter, it must be contained in a tuple. If the method has no
-- out-parameters, /@parameters@/ may be 'P.Nothing' or an empty tuple.
-- 
-- 
-- === /C code/
-- >
-- >GDBusMethodInvocation *invocation = some_invocation;
-- >g_autofree gchar *result_string = NULL;
-- >g_autoptr (GError) error = NULL;
-- >
-- >result_string = calculate_result (&error);
-- >
-- >if (error != NULL)
-- >  g_dbus_method_invocation_return_gerror (invocation, error);
-- >else
-- >  g_dbus_method_invocation_return_value (invocation,
-- >                                         g_variant_new ("(s)", result_string));
-- >
-- >// Do not free @invocation here; returning a value does that
-- 
-- 
-- This method will take ownership of /@invocation@/. See
-- t'GI.Gio.Structs.DBusInterfaceVTable.DBusInterfaceVTable' for more information about the ownership of
-- /@invocation@/.
-- 
-- Since 2.48, if the method call requested for a reply not to be sent
-- then this call will sink /@parameters@/ and free /@invocation@/, but
-- otherwise do nothing (as per the recommendations of the D-Bus
-- specification).
-- 
-- /Since: 2.26/
dBusMethodInvocationReturnValue ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> Maybe (GVariant)
    -- ^ /@parameters@/: A t'GVariant' tuple with out parameters for the method or 'P.Nothing' if not passing any parameters.
    -> m ()
dBusMethodInvocationReturnValue invocation parameters = liftIO $ do
    invocation' <- B.ManagedPtr.disownObject invocation
    maybeParameters <- case parameters of
        Nothing -> return FP.nullPtr
        Just jParameters -> do
            jParameters' <- unsafeManagedPtrGetPtr jParameters
            return jParameters'
    g_dbus_method_invocation_return_value invocation' maybeParameters
    touchManagedPtr invocation
    whenJust parameters touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationReturnValueMethodInfo
instance (signature ~ (Maybe (GVariant) -> m ()), MonadIO m, IsDBusMethodInvocation a) => O.OverloadedMethod DBusMethodInvocationReturnValueMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationReturnValue

instance O.OverloadedMethodInfo DBusMethodInvocationReturnValueMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationReturnValue",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationReturnValue"
        })


#endif

-- method DBusMethodInvocation::return_value_with_unix_fd_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "invocation"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "DBusMethodInvocation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GDBusMethodInvocation."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "parameters"
--           , argType = TVariant
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "A #GVariant tuple with out parameters for the method or %NULL if not passing any parameters."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "fd_list"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "UnixFDList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GUnixFDList or %NULL."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_dbus_method_invocation_return_value_with_unix_fd_list" g_dbus_method_invocation_return_value_with_unix_fd_list :: 
    Ptr DBusMethodInvocation ->             -- invocation : TInterface (Name {namespace = "Gio", name = "DBusMethodInvocation"})
    Ptr GVariant ->                         -- parameters : TVariant
    Ptr Gio.UnixFDList.UnixFDList ->        -- fd_list : TInterface (Name {namespace = "Gio", name = "UnixFDList"})
    IO ()

-- | Like 'GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationReturnValue' but also takes a t'GI.Gio.Objects.UnixFDList.UnixFDList'.
-- 
-- This method is only available on UNIX.
-- 
-- This method will take ownership of /@invocation@/. See
-- t'GI.Gio.Structs.DBusInterfaceVTable.DBusInterfaceVTable' for more information about the ownership of
-- /@invocation@/.
-- 
-- /Since: 2.30/
dBusMethodInvocationReturnValueWithUnixFdList ::
    (B.CallStack.HasCallStack, MonadIO m, IsDBusMethodInvocation a, Gio.UnixFDList.IsUnixFDList b) =>
    a
    -- ^ /@invocation@/: A t'GI.Gio.Objects.DBusMethodInvocation.DBusMethodInvocation'.
    -> Maybe (GVariant)
    -- ^ /@parameters@/: A t'GVariant' tuple with out parameters for the method or 'P.Nothing' if not passing any parameters.
    -> Maybe (b)
    -- ^ /@fdList@/: A t'GI.Gio.Objects.UnixFDList.UnixFDList' or 'P.Nothing'.
    -> m ()
dBusMethodInvocationReturnValueWithUnixFdList invocation parameters fdList = liftIO $ do
    invocation' <- B.ManagedPtr.disownObject invocation
    maybeParameters <- case parameters of
        Nothing -> return FP.nullPtr
        Just jParameters -> do
            jParameters' <- unsafeManagedPtrGetPtr jParameters
            return jParameters'
    maybeFdList <- case fdList of
        Nothing -> return FP.nullPtr
        Just jFdList -> do
            jFdList' <- unsafeManagedPtrCastPtr jFdList
            return jFdList'
    g_dbus_method_invocation_return_value_with_unix_fd_list invocation' maybeParameters maybeFdList
    touchManagedPtr invocation
    whenJust parameters touchManagedPtr
    whenJust fdList touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data DBusMethodInvocationReturnValueWithUnixFdListMethodInfo
instance (signature ~ (Maybe (GVariant) -> Maybe (b) -> m ()), MonadIO m, IsDBusMethodInvocation a, Gio.UnixFDList.IsUnixFDList b) => O.OverloadedMethod DBusMethodInvocationReturnValueWithUnixFdListMethodInfo a signature where
    overloadedMethod = dBusMethodInvocationReturnValueWithUnixFdList

instance O.OverloadedMethodInfo DBusMethodInvocationReturnValueWithUnixFdListMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.DBusMethodInvocation.dBusMethodInvocationReturnValueWithUnixFdList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-DBusMethodInvocation.html#v:dBusMethodInvocationReturnValueWithUnixFdList"
        })


#endif


