/*
 * Copyright (C) 2014-2025 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QLabel>
#include <QMap>
#include <QSet>

#include "src/datovka_shared/identifiers/account_id.h" /* AcntId */

class AccountsMap; /* Forward declaration. */
class DbContainer; /* Forward declaration. */
class QString; /* Forward declaration. */
class QTextEdit; /* Forward declaration. */

/*!
 * @brief Message database status control widget.
 */
class DbStatusControl : public QLabel {
	Q_OBJECT

public:
	/*!
	 * @brief Disk or memory database location.
	 */
	enum DbLocation {
		LOC_DISK = 0, /*!< Database on disk. */
		LOC_MEMORY /*!< Database in memory. */
	};

	/*!
	 * @brief Constructor.
	 *
	 * @param[in] regularAccounts Regular account container.
	 * @param[in] dbCont Database set container.
	 * @param[in] parent Parent widget.
	 */
	explicit DbStatusControl(AccountsMap *regularAccounts,
	    DbContainer *dbCont, QWidget *parent = Q_NULLPTR);

	/*!
	 * @brief Destructor.
	 */
	~DbStatusControl(void);

	void setStorageLocations(enum DbLocation msgDbLoc,
	    enum DbLocation acntDbLoc);

protected:
	/*!
	 * @brief Raises status list widget.
	 *
	 * @param[in] event Mouse press event.
	 */
	virtual
	void mousePressEvent(QMouseEvent *event) Q_DECL_OVERRIDE;

private Q_SLOTS:
	/*!
	 * @brief Update the content after account data has been modified.
	 *
	 * @param[in] acntId Account identifier.
	 */
	void watchAccountDataChanged(const AcntId &acntId);

	/*!
	 * @brief Display a status bar notification that database file cannot
	 *     be opened.
	 *
	 * @param[in] acntId Account identifier.
	 * @param[in] fileName Database file name.
	 */
	void watchOpeningFailed(const AcntId &acntId, const QString &fileName);

private:
	/*!
	 * @brief Raise the pop-up widget.
	 */
	void raisePopup(void);

	AccountsMap *m_accounts; /*!< Pointer to account container (regular). */

	QMap<AcntId, QSet<QString> > m_problemDbFiles; /*!< Paths to problematic files. */

	QWidget *m_dbPopup; /*!< Database status pop-up. */
	QTextEdit *m_textEdit; /*!< Displays error message. */

	enum DbLocation m_msgDbLoc; /*!< Message database location. */
	enum DbLocation m_acntDbLoc; /*!< Account database location. */
};
